using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using LicenseSpring;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

public class LicenseDetailsUI : MonoBehaviour
{
    private ILicense license => LicenseManager.GetInstance().CurrentLicense();

    [Header("UI References")]
    public TMP_Text keyText;
    public TMP_Text typeText;
    public TMP_Text statusText;
    public TMP_Text isOfflineActivatedText;
    public TMP_Text isValidText;
    public TMP_Text isTrialText;
    public TMP_Text validityPeriodText;
    public TMP_Text startDateText;
    public TMP_Text daysRemainingText;
    public TMP_Text subscriptionGracePeriodText;
    public TMP_Text maintenancePeriodText;
    public TMP_Text lastCheckedText;
    public TMP_Text isFloatingText;
    public TMP_Text floatingTimeoutText;
    public TMP_Text totalConsumptionText;
    public TMP_Text maxConsumptionText;
    public Button syncConsumptionButton;
    public Button updateConsumptionButton;
    public TMP_InputField updateConsumptionValue;
    public Button checkLicenseButton;
    public Button borrowLicenseButton;
    public TMP_Text deviceVariablesText;
    public Button addDeviceVariableButton;

    // Start is called once before the first execution of Update after the MonoBehaviour is created
    void Start()
    {
        StartCoroutine(WaitForInitialization());
    }

    private IEnumerator WaitForInitialization()
    {
        while (LicenseSpringManager.Instance == null || !LicenseSpringManager.Instance.IsInitialized)
        {
            yield return null;
        }
        InitializeUI();
        UpdateLicenseInfo();
    }

    private void InitializeUI()
    {
        syncConsumptionButton.onClick.AddListener(OnSyncConsumptionClicked);
        updateConsumptionButton.onClick.AddListener(OnUpdateConsumptionClicked);
        checkLicenseButton.onClick.AddListener(OnCheckLicenseClicked);
        borrowLicenseButton.onClick.AddListener(OnBorrowLicenseClicked);
        addDeviceVariableButton.onClick.AddListener(OnAddDeviceVariableClicked);
    }

    private void OnBorrowLicenseClicked()
    {
        try
        {
            if (license == null)
            {
                Debug.Log("No license detected.");
                return;
            }

            if (license.IsBorrowed())
            {
                license.Release();
                Debug.Log("Borrowed license released.");
            }
            else
            {
                license.Borrow();
                if (license.IsBorrowed())
                {
                    string borrowedUntil = license.FloatingEndDateTime().ToString();
                    Debug.Log($"License borrowed until {borrowedUntil}.");
                }
                else
                {
                    Debug.Log("License failed to borrow.");
                }
            }
            UpdateLicenseInfo();
        }
        catch (Exception ex)
        {
            Debug.LogError($"Exception occured while trying to borrow license: {ex.Message}");
        }
    }

    private void OnCheckLicenseClicked()
    {
        if (license == null)
        {
            Debug.Log("No license detected.");
            return;
        }

        try
        {
            license.Check();
            Debug.Log("License successfully checked.");
            UpdateLicenseInfo();
        }
        catch (Exception ex)
        {
            Debug.LogError($"Exception occured during license check: {ex.Message}");
        }
    }

    private void OnUpdateConsumptionClicked()
    {
        if (license == null)
        {
            Debug.Log("No license detected.");
            return;
        }

        if (int.TryParse(updateConsumptionValue.text, out int consumptionValue))
        {
            try
            {
                license.UpdateConsumption(consumptionValue);
                UpdateConsumptionUI();
            }
            catch (Exception ex)
            {
                Debug.LogError($"Exception while updating consumption: {ex.Message}");
            }
        }
    }

    private void OnSyncConsumptionClicked()
    {
        if (license == null)
        {
            Debug.Log("No license detected.");
            return;
        }

        try
        {
            license.SyncConsumption();
            UpdateConsumptionUI();
        }
        catch (Exception ex)
        {
            Debug.LogError($"Exception while syncing consumption: {ex.Message}");
        }
    }

    private void OnAddDeviceVariableClicked()
    {
        string newVariableName = RandomString();
        string newVariableValue = RandomString();
        license.AddDeviceVariable(newVariableName, newVariableValue);
        license.SendDeviceVariables();
        UpdateLicenseInfo();
    }

    private string RandomString()
    {
        string allowedChars = "abcdefghijkmnopqrstuvwxyz";
        System.Random r = new System.Random();
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < 3; i++)
        {
            int index = r.Next(0, allowedChars.Length);
            char randomChar = allowedChars[index];
            sb.Append(randomChar);
        }
        return sb.ToString();
    }

    private void UpdateConsumptionUI()
    {
        if (license.IsUnlimitedConsumptionAllowed())
        {
            totalConsumptionText.text = "Unlimited";
        }
        else
        {
            totalConsumptionText.text = license.TotalConsumption().ToString();
        }
    }

    public void UpdateLicenseInfo()
    {
        try
        {
            if (license == null || !license.Status().IsActive())
            {
                keyText.text = "";
                typeText.text = "";
                statusText.text = "";
                isOfflineActivatedText.text = "";
                isValidText.text = "";
                isTrialText.text = "";
                validityPeriodText.text = "";
                startDateText.text = "";
                daysRemainingText.text = "";
                subscriptionGracePeriodText.text = "";
                maintenancePeriodText.text = "";
                lastCheckedText.text = "";
                isFloatingText.text = "";
                floatingTimeoutText.text = "";
                totalConsumptionText.text = "";
                maxConsumptionText.text = "";
                deviceVariablesText.text = "";
            }
            else
            {
                keyText.text = license.Key();
                typeText.text = license.Type().ToString();
                statusText.text = license.Status().ToString();
                isOfflineActivatedText.text = license.IsOfflineActivated().ToString();
                isValidText.text = license.IsValid().ToString();
                isTrialText.text = license.IsTrial().ToString();
                validityPeriodText.text = license.ValidityPeriod().ToString("dd/MM/yyyy");
                startDateText.text = license.StartDate().ToString("dd/MM/yyyy");
                daysRemainingText.text = license.DaysRemaining().ToString();
                subscriptionGracePeriodText.text = license.Type() == LicenseType.Subscription ? license.SubscriptionGracePeriod().ToString() : "-";
                maintenancePeriodText.text = license.MaintenancePeriod().ToString("dd/MM/yyyy");
                lastCheckedText.text = license.LastCheckDate().ToString();
                isFloatingText.text = license.IsFloating().ToString();
                floatingTimeoutText.text = license.FloatingTimeout().ToString();
                totalConsumptionText.text = license.TotalConsumption().ToString();
                maxConsumptionText.text = license.MaxConsumption().ToString();

                List<DeviceVariable> deviceVariablesFromBackend = license.GetDeviceVariables(getFromBackend: true).ToList();

                if (deviceVariablesFromBackend.Count == 0)
                {
                    deviceVariablesText.text = "";
                }
                else
                {
                    deviceVariablesText.text = string.Join("; ", deviceVariablesFromBackend.Select(e => $"{e.Name}:{e.Value}").ToList());
                }
            }
        }
        catch (System.Exception ex)
        {
            Debug.LogError("Failed to update license information: " + ex.Message);
        }
    }
}
