using UnityEngine;
using TMPro;
using UnityEngine.UI;
using LicenseSpring;
using System.IO;

public class LicenseManagementUI : MonoBehaviour
{
    [Header("UI References")]
    public TMP_InputField licenseKeyInput;
    public TMP_Text localFilesPathText;
    public Button activateButton;
    public Button deactivateButton;
    public Button deleteFilesButton;
    public LicenseDetailsUI licenseDetailsUI;

    private LicenseSpringManager licenseManager;

    // Start is called once before the first execution of Update after the MonoBehaviour is created
    void Start()
    {
        StartCoroutine(WaitForInitialization());
    }
    private System.Collections.IEnumerator WaitForInitialization()
    {
        while (LicenseSpringManager.Instance == null || !LicenseSpringManager.Instance.IsInitialized)
            yield return null;

        licenseManager = LicenseSpringManager.Instance;
        InitializeUI();
    }
    private void InitializeUI()
    {
        // Set button listeners
        activateButton.onClick.AddListener(OnActivateClicked);
        deactivateButton.onClick.AddListener(OnDeactivateClicked);
        deleteFilesButton.onClick.AddListener(OnDeleteFilesClicked);

        UpdateLocalFilesPath();
        licenseDetailsUI?.UpdateLicenseInfo();
    }

    private void OnActivateClicked()
    {
        string licenseKey = licenseKeyInput.text?.Trim();

        if (string.IsNullOrEmpty(licenseKey))
        {
            Debug.LogWarning("Please enter a license key before activating.");
            return;
        }

        try
        {
            ILicense license = licenseManager.ActivateLicense(licenseKey);
            Debug.Log("License activated successfully!");
            UpdateLocalFilesPath();
            licenseDetailsUI?.UpdateLicenseInfo();
        }
        catch (System.Exception ex)
        {
            Debug.LogError("Failed to activate license: " + ex.Message);
        }
    }

    private void OnDeactivateClicked()
    {
        try
        {
            bool success = licenseManager.DeactivateCurrentLicense();

            if (success)
            {
                Debug.Log("License deactivated successfully!");
                UpdateLocalFilesPath();
                licenseDetailsUI?.UpdateLicenseInfo();
            }
        }
        catch (System.Exception ex)
        {
            Debug.LogError("Failed to deactivate license: " + ex.Message);
        }
    }

    private void OnDeleteFilesClicked()
    {
        try
        {
            licenseManager.ClearLocalStorage();
            Debug.Log("Local license files deleted.");
            UpdateLocalFilesPath();
        }
        catch (System.Exception ex)
        {
            Debug.LogError("Failed to delete local files: " + ex.Message);
        }
    }

    private void UpdateLocalFilesPath()
    {
        try
        {
            string dataLocation = licenseManager.GetDataLocation();
            string licenseFileName = licenseManager.GetLicenseFileName();
            string fullFilePath = Path.GetFullPath(Path.Combine(dataLocation, licenseFileName));

            if (File.Exists(fullFilePath))
            {
                localFilesPathText.text = fullFilePath;
                Debug.Log("Local license file path: " + fullFilePath);
            }
            else
            {
                localFilesPathText.text = "";
            }
        }
        catch (System.Exception ex)
        {
            localFilesPathText.text = "Unavailable";
            Debug.LogWarning("Could not determine local file path: " + ex.Message);
        }
    }
}
