using System;
using LicenseSpring;
using UnityEngine;

public class LicenseSpringManager : MonoBehaviour
{
    public static LicenseSpringManager Instance { get; private set; }
    public bool IsInitialized { get; private set; }


    private void Awake()
    {
        if (Instance != null && Instance != this)
        {
            Destroy(gameObject);
            return;
        }
        Instance = this;
        InitializeSdk();
    }


    private void InitializeSdk()
    {
        if (IsInitialized)
        {
            return;
        }
        
        try
        {
            Debug.Log("Initializing LicenseSpring SDK...");

            Configuration configuration = LSConfig.CreateConfiguration();
            ILicenseManager manager = LicenseManager.GetInstance();
            manager.Initialize(configuration);
            IsInitialized = true;

            Debug.Log("LicenseSpring SDK successfully initialized!");
            Debug.Log($"HWID: {configuration.HardwareID}");
        }
        catch (Exception ex)
        {
            Debug.LogError($"LicenseSpring initialization failed: {ex.Message}");
        }
    }

    public ILicense ActivateLicense(string licenseKey)
    {
        LicenseID id = LicenseID.FromKey(licenseKey);
        ILicense license = LicenseManager.GetInstance().ActivateLicense(id);
        return license;
    }

    public bool DeactivateCurrentLicense()
    {
        ILicense license = LicenseManager.GetInstance().CurrentLicense();
        if (license == null)
        {
            Debug.LogWarning("No active license to deactivate.");
            return false;
        }
        license.Deactivate();
        return true;
    }

    internal void ClearLocalStorage() => LicenseManager.GetInstance().ClearLocalStorage();

    internal string GetDataLocation() => LicenseManager.GetInstance().DataLocation;

    internal string GetLicenseFileName() => LicenseManager.GetInstance().LicenseFileName;
}
