using System;
using System.Collections.Generic;
using System.Linq;
using LicenseSpring;

namespace BundleSample
{
    /// <summary>
    /// Demonstrates bundle license functionality in the LicenseSpring .NET SDK.
    /// A bundle allows activating multiple product licenses together as a single unit.
    /// </summary>
    class Program
    {
        private static BundleManager _bundleManager;
        private static Configuration _configuration;

        static void Main(string[] args)
        {
            Console.OutputEncoding = System.Text.Encoding.UTF8;
            Console.WriteLine("===================================");
            Console.WriteLine("LicenseSpring Bundle License Sample");
            Console.WriteLine("===================================\n");

            try
            {
                // Initialize configuration
                _configuration = LSConfig.CreateConfiguration();

                // Display SDK information
                DisplaySDKInfo();

                // Initialize BundleManager
                _bundleManager = (BundleManager)BundleManager.GetInstance();
                _bundleManager.Initialize(_configuration);
                Console.WriteLine("BundleManager initialized successfully.\n");

                // Main menu
                bool exit = false;
                while (!exit)
                {
                    DisplayMenu();
                    string choice = Console.ReadLine()?.Trim();

                    switch (choice)
                    {
                        case "1":
                            ActivateBundleOnline();
                            break;
                        case "2":
                            ViewCurrentBundle();
                            break;
                        case "3":
                            CheckBundle();
                            break;
                        case "4":
                            DeactivateBundleOnline();
                            break;
                        case "5":
                            OfflineActivationWorkflow();
                            break;
                        case "6":
                            OfflineDeactivationWorkflow();
                            break;
                        case "7":
                            TestErrorHandling();
                            break;
                        case "8":
                            ClearLocalStorage();
                            break;
                        case "9":
                            exit = true;
                            Console.WriteLine("Exiting demo.");
                            break;
                        default:
                            Console.WriteLine("Invalid choice. Please try again.\n");
                            break;
                    }
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }

            Console.WriteLine("\nPress any key to exit...");
            Console.ReadKey();
        }

        static void DisplaySDKInfo()
        {
            Console.WriteLine("------------- SDK Information -------------");
            Console.WriteLine($"Application:             {_configuration.AppName} {_configuration.AppVersion}");
            Console.WriteLine($"LicenseSpring SDK:       {_configuration.SdkVersion}");
            Console.WriteLine($"Product Code:            {_configuration.ProductCode}");
            Console.WriteLine($"OS Version:              {_configuration.OSVersion}");
            Console.WriteLine($"Hardware ID Algorithm:   {_configuration.ExtendedOptions.DeviceIdAlgorithm}");
            Console.WriteLine($"Hardware ID:             {_configuration.HardwareID}");
            Console.WriteLine();
        }

        static void DisplayMenu()
        {
            Console.WriteLine("\n========================================");
            Console.WriteLine("Bundle License Operations");
            Console.WriteLine("========================================");
            Console.WriteLine("1. Activate bundle (online)");
            Console.WriteLine("2. View current bundle");
            Console.WriteLine("3. Check bundle (sync with server)");
            Console.WriteLine("4. Deactivate bundle (online)");
            Console.WriteLine("5. Offline activation workflow");
            Console.WriteLine("6. Offline deactivation workflow");
            Console.WriteLine("7. Test error handling");
            Console.WriteLine("8. Clear local storage");
            Console.WriteLine("9. Exit");
            Console.WriteLine("========================================");
            Console.Write("Enter your choice: ");
        }

        #region Online Operations

        static void ActivateBundleOnline()
        {
            Console.WriteLine("\n--- Online Bundle Activation ---");
            Console.Write("Enter bundle license key: ");
            string licenseKey = Console.ReadLine()?.Trim();

            if (string.IsNullOrEmpty(licenseKey))
            {
                Console.WriteLine("Note: You need to provide a valid bundle license key.");
                Console.WriteLine("Create a bundle license in the LicenseSpring platform first.");
                return;
            }

            try
            {
                Console.WriteLine("Activating bundle...");
                Dictionary<string, ILicense> bundle = _bundleManager.ActivateBundle(LicenseID.FromKey(licenseKey));

                Console.WriteLine($"\nBundle activated successfully with {bundle.Count} product(s):");
                // Note you have one more license in the bundle, for the bundle itself.
                foreach (KeyValuePair<string, ILicense> kvp in bundle)
                {
                    Console.WriteLine($"\n  Product Code: {kvp.Key}");
                    PrintLicenseInfo(kvp.Value, "    ");
                }

                Console.WriteLine("\nBundle activation complete.");
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        static void ViewCurrentBundle()
        {
            Console.WriteLine("\n--- Current Bundle ---");

            try
            {
                Dictionary<string, ILicense> bundle = _bundleManager.GetCurrentBundle();

                if (bundle == null || bundle.Count == 0)
                {
                    Console.WriteLine("No bundle found in local storage.");
                    Console.WriteLine("Please activate a bundle first (option 1).");
                    return;
                }

                Console.WriteLine($"Bundle contains {bundle.Count} product(s):\n");
                foreach (KeyValuePair<string, ILicense> kvp in bundle)
                {
                    Console.WriteLine($"Product Code: {kvp.Key}");
                    Console.WriteLine(new string('-', 40));
                    PrintLicenseInfo(kvp.Value, "  ");
                    Console.WriteLine();
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        static void CheckBundle()
        {
            Console.WriteLine("\n--- Check Bundle (Sync with Server) ---");

            try
            {
                Dictionary<string, ILicense> bundle = _bundleManager.GetCurrentBundle();
                if (bundle == null || bundle.Count == 0)
                {
                    Console.WriteLine("No bundle found. Please activate a bundle first.");
                    return;
                }

                // Get license key from first license in bundle
                ILicense firstLicense = bundle.Values.First();
                string licenseKey = firstLicense.Id().Key;

                Console.WriteLine("Synchronizing bundle with server...");
                _bundleManager.CheckBundle(LicenseID.FromKey(licenseKey));

                Console.WriteLine("Bundle checked successfully.");
                Console.WriteLine("\nUpdated bundle information:");
                ViewCurrentBundle();
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        static void DeactivateBundleOnline()
        {
            Console.WriteLine("\n--- Online Bundle Deactivation ---");

            try
            {
                Dictionary<string, ILicense> bundle = _bundleManager.GetCurrentBundle();
                if (bundle == null || bundle.Count == 0)
                {
                    Console.WriteLine("No bundle found. Please activate a bundle first.");
                    return;
                }

                // Get license key from first license in bundle
                ILicense firstLicense = bundle.Values.First();
                string licenseKey = firstLicense.Id().Key;

                Console.Write("Remove local data after deactivation? (y/n): ");
                string removeChoice = Console.ReadLine()?.Trim().ToLower();
                bool removeLocalData = removeChoice == "y";

                Console.WriteLine("Deactivating bundle...");
                bool success = _bundleManager.DeactivateBundle(LicenseID.FromKey(licenseKey), removeLocalData);

                if (success)
                {
                    Console.WriteLine("Bundle deactivated successfully.");
                    if (removeLocalData)
                        Console.WriteLine("  Local data has been removed.");
                    else
                        Console.WriteLine("  Local data preserved.");
                }
                else
                {
                    Console.WriteLine("Bundle deactivation failed.");
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        #endregion

        #region Offline Operations

        static void OfflineActivationWorkflow()
        {
            Console.WriteLine("\n--- Offline Bundle Activation Workflow ---");

            Console.WriteLine("Choose an option:");
            Console.WriteLine("1. Generate activation request file");
            Console.WriteLine("2. Activate from response file");
            Console.Write("Enter choice: ");
            string choice = Console.ReadLine()?.Trim();

            if (choice == "1")
            {
                GenerateOfflineActivationRequest();
            }
            else if (choice == "2")
            {
                ActivateFromOfflineResponse();
            }
            else
            {
                Console.WriteLine("Invalid choice.");
            }
        }

        static void GenerateOfflineActivationRequest()
        {
            Console.WriteLine("\n--- Generate Offline Activation Request ---");
            Console.Write("Enter bundle license key: ");
            string licenseKey = Console.ReadLine()?.Trim();

            if (string.IsNullOrEmpty(licenseKey))
            {
                Console.WriteLine("License key cannot be empty.");
                return;
            }

            Console.Write("Enter output file path (or press Enter for default): ");
            string filePath = Console.ReadLine()?.Trim();
            if (string.IsNullOrEmpty(filePath))
            {
                filePath = null; // Use default path
            }

            try
            {
                Console.WriteLine("Generating activation request file...");
                string outputPath = _bundleManager.GetOfflineActivationFile(
                    licenseID: LicenseID.FromKey(licenseKey),
                    deviceVariables: null,
                    activationRequestFile: filePath);

                Console.WriteLine($"Activation request file created: {outputPath}");
                Console.WriteLine("\nNext steps:");
                Console.WriteLine("1. Upload this file to https://offline.licensespring.com");
                Console.WriteLine("2. Download the activation response file");
                Console.WriteLine("3. Use option 5 -> 2 to activate from response file");
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        static void ActivateFromOfflineResponse()
        {
            Console.WriteLine("\n--- Activate from Offline Response ---");
            Console.Write("Enter response file path (or press Enter for default): ");
            string filePath = Console.ReadLine()?.Trim();

            if (string.IsNullOrEmpty(filePath))
            {
                filePath = null; // Use default path
            }

            try
            {
                Console.WriteLine("Activating bundle from offline response...");
                Dictionary<string, ILicense> bundle = _bundleManager.ActivateBundleOffline(filePath);

                Console.WriteLine($"\nBundle activated successfully with {bundle.Count} product(s):");
                foreach (KeyValuePair<string, ILicense> kvp in bundle)
                {
                    Console.WriteLine($"\n  Product Code: {kvp.Key}");
                    PrintLicenseInfo(kvp.Value, "    ");
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        static void OfflineDeactivationWorkflow()
        {
            Console.WriteLine("\n--- Offline Bundle Deactivation Workflow ---");

            try
            {
                Dictionary<string, ILicense> bundle = _bundleManager.GetCurrentBundle();
                if (bundle == null || bundle.Count == 0)
                {
                    Console.WriteLine("No bundle found. Please activate a bundle first.");
                    return;
                }

                // Get license key from first license in bundle
                ILicense firstLicense = bundle.Values.First();
                string licenseKey = firstLicense.Id().Key;

                Console.Write("Enter output file path (or press Enter for default): ");
                string filePath = Console.ReadLine()?.Trim();
                if (string.IsNullOrEmpty(filePath))
                {
                    filePath = null; // Use default path
                }

                Console.WriteLine("Generating deactivation request file...");
                string outputPath = _bundleManager.DeactivateBundleOffline(
                    LicenseID.FromKey(licenseKey),
                    deactivationRequestFile: filePath);

                Console.WriteLine($"Deactivation request file created: {outputPath}");
                Console.WriteLine("\nNext steps:");
                Console.WriteLine("1. Upload this file to https://offline.licensespring.com");
                Console.WriteLine("2. The activation slot will be freed on the server");
                Console.WriteLine("3. Clear local storage (option 8) to remove local license data");
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        #endregion

        #region Error Handling

        static void TestErrorHandling()
        {
            Console.WriteLine("\n--- Error Handling Tests ---");
            Console.WriteLine("These tests demonstrate error handling scenarios.\n");

            Console.WriteLine("1. Invalid license key");
            Console.WriteLine("2. Network failure simulation");
            Console.WriteLine("3. Activate non-bundle license as bundle");
            Console.Write("Enter choice: ");
            string choice = Console.ReadLine()?.Trim();

            switch (choice)
            {
                case "1":
                    TestInvalidLicenseKey();
                    break;
                case "2":
                    TestNetworkFailure();
                    break;
                case "3":
                    TestNonBundleLicense();
                    break;
                default:
                    Console.WriteLine("Invalid choice.");
                    break;
            }
        }

        static void TestInvalidLicenseKey()
        {
            Console.WriteLine("\n--- Test: Invalid License Key ---");
            try
            {
                Console.WriteLine("Attempting to activate bundle with invalid key...");
                Dictionary<string, ILicense> bundle = _bundleManager.ActivateBundle(LicenseID.FromKey("INVALID-KEY-1234"));
                Console.WriteLine("Unexpected: Activation succeeded (should have failed).");
            }
            catch (LicenseNotFoundException ex)
            {
                Console.WriteLine("Expected exception caught:");
                Console.WriteLine($"  {ex.GetType().Name}: {ex.Message}");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Unexpected exception:");
                PrintException(ex);
            }
        }

        static void TestNetworkFailure()
        {
            Console.WriteLine("\n--- Test: Network Failure ---");
            Console.WriteLine("This test requires disconnecting from the internet.");
            Console.Write("Are you offline? (y/n): ");
            string offline = Console.ReadLine()?.Trim().ToLower();

            if (offline != "y")
            {
                Console.WriteLine("Test skipped. Disconnect from the internet and try again.");
                return;
            }

            try
            {
                Console.WriteLine("Attempting to activate bundle while offline...");
                Dictionary<string, ILicense> bundle = _bundleManager.ActivateBundle(LicenseID.FromKey("TEST-KEY-1234"));
                Console.WriteLine("Unexpected: Activation succeeded (should have failed).");
            }
            catch (NetworkException ex)
            {
                Console.WriteLine("Expected exception caught:");
                Console.WriteLine($"  {ex.GetType().Name}: {ex.Message}");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Caught exception:");
                PrintException(ex);
            }
        }

        static void TestNonBundleLicense()
        {
            Console.WriteLine("\n--- Test: Non-Bundle License ---");
            Console.Write("Enter a regular (non-bundle) license key to test: ");
            string licenseKey = Console.ReadLine()?.Trim();

            if (string.IsNullOrEmpty(licenseKey))
            {
                Console.WriteLine("License key cannot be empty.");
                return;
            }

            try
            {
                Console.WriteLine("Attempting to activate non-bundle license as bundle...");
                Dictionary<string, ILicense> bundle = _bundleManager.ActivateBundle(LicenseID.FromKey(licenseKey));
                Console.WriteLine("Activation succeeded (expected failure for non-bundle license).");
            }
            catch (LicenseNotFoundException ex)
            {
                Console.WriteLine("Expected exception caught:");
                Console.WriteLine($"  {ex.GetType().Name}: {ex.Message}");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Caught exception:");
                PrintException(ex);
            }
        }

        #endregion

        #region Helper Methods

        static void ClearLocalStorage()
        {
            Console.WriteLine("\n--- Clear Local Storage ---");
            Console.Write("Are you sure you want to clear all local license data? (y/n): ");
            string confirm = Console.ReadLine()?.Trim().ToLower();

            if (confirm == "y")
            {
                try
                {
                    _bundleManager.ClearLocalStorage();
                    Console.WriteLine("Local storage cleared successfully.");
                }
                catch (Exception ex)
                {
                    PrintException(ex);
                }
            }
            else
            {
                Console.WriteLine("Operation cancelled.");
            }
        }

        static void PrintLicenseInfo(ILicense license, string indent = "")
        {
            if (license == null)
            {
                Console.WriteLine($"{indent}(null license)");
                return;
            }

            try
            {
                Console.WriteLine($"{indent}License Key:      {license.Id().Key}");
                Console.WriteLine($"{indent}Status:           {(license.Status().IsActive() ? "Active" : "Inactive")}");
                Console.WriteLine($"{indent}License Type:     {license.Type()}");

                DateTime validityPeriod = license.ValidityPeriod();
                if (validityPeriod != DateTime.MinValue && validityPeriod.Year > 2000)
                {
                    Console.WriteLine($"{indent}Valid Until:      {validityPeriod:yyyy-MM-dd HH:mm:ss}");
                }

                Console.WriteLine($"{indent}Max Activations:  {license.MaxActivations()}");
                Console.WriteLine($"{indent}Times Activated:  {license.TimesActivated()}");

                // Display features
                LicenseFeature[] features = license.Features();
                if (features != null && features.Length > 0)
                {
                    Console.WriteLine($"{indent}Features ({features.Length}):");
                    foreach (LicenseFeature feature in features)
                    {
                        string featureInfo = $"{indent}  - {feature.Code}: {feature.Name}";
                        if (feature.FeatureType == LicenseFeature.Type.Consumption)
                        {
                            featureInfo += $" (Consumption: {feature.TotalConsumption}/{feature.MaxConsumption})";
                        }
                        Console.WriteLine(featureInfo);
                    }
                }

                // Display consumption info if applicable
                if (license.Type() == LicenseType.Consumption)
                {
                    Console.WriteLine($"{indent}Consumption:      {license.TotalConsumption()}/{license.MaxConsumption()}");
                }

                // Display custom fields
                CustomField[] customFields = license.CustomFields();
                if (customFields != null && customFields.Length > 0)
                {
                    Console.WriteLine($"{indent}Custom Fields ({customFields.Length}):");
                    foreach (var field in customFields)
                    {
                        Console.WriteLine($"{indent}  - {field.Name}: {field.Value}");
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"{indent}Error reading license info: {ex.Message}");
            }
        }

        static void PrintException(Exception ex)
        {
            Console.WriteLine("\nException occurred:");
            Console.WriteLine($"  Type:    {ex.GetType().Name}");
            Console.WriteLine($"  Message: {ex.Message}");

            if (ex.InnerException != null)
            {
                Console.WriteLine($"  Inner:   {ex.InnerException.GetType().Name}: {ex.InnerException.Message}");
            }

            // OAuthException has ErrorCode property
            if (ex is OAuthException oauthEx && !string.IsNullOrEmpty(oauthEx.ErrorCode))
            {
                Console.WriteLine($"  Code:    {oauthEx.ErrorCode}");
            }

            // LicenseActivationException has Code property
            if (ex is LicenseActivationException activationEx && !string.IsNullOrEmpty(activationEx.Code))
            {
                Console.WriteLine($"  Code:    {activationEx.Code}");
            }
        }

        #endregion
    }
}