﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using LicenseSpring;

namespace Sample.LicenseSpring
{
    static class Program
    {
        public static void Main( string[] args )
        {
            Console.OutputEncoding = System.Text.Encoding.UTF8;

            var configuration = LSConfig.CreateConfiguration();

            Console.WriteLine( "------------- General info -------------" );
            Console.WriteLine( configuration.AppName + ' ' + configuration.AppVersion );
            Console.WriteLine( "LicenseSpring SDK version: " + configuration.SdkVersion );
            Console.WriteLine( "LicenseSpring API version: " + configuration.LicenseSpringAPIVersion );
            Console.WriteLine( "Determined OS version:     " + configuration.OSVersion );
            Console.WriteLine();

            // Detect virtualized environment (VM detection by default disabled)
            if( configuration.IsVMDetectionEnabled )
            {
                Console.WriteLine( "Checking for virtual machines..." );
                string msg;
                if( configuration.IsVM )
                {
                    msg = "Virtual machine detected!";
                    if( !string.IsNullOrEmpty( configuration.DetectedVMName ) )
                        msg += " Hypervisor name: " + configuration.DetectedVMName;
                }
                else
                    msg = "Check passed, VM not detected.";
                Console.WriteLine( msg );
                Console.WriteLine();
            }

            Console.WriteLine( "------------- Network info -------------" );
            Console.WriteLine( "Host name:   " + configuration.Hostname );
            Console.WriteLine( "Local IP:    " + configuration.LocalIp );
            Console.WriteLine( "MAC address: " + configuration.MACAddress );
            Console.WriteLine();

            bool deactivateAndRemove = false; // deactivate and remove license after sample execution

            try
            {
                // Uncomment next line to test offline Floating server
                //TestFloatingServer( configuration ); return;
                var licenseManager = LicenseManager.GetInstance();
                licenseManager.Initialize( configuration );
                // Use the initialization below if you need to synchronize access to local license between process
                //licenseManager.Initialize( configuration, new LicenseFileStorageEx() );

                // Get basic information about configured product. Include optional parameters includeCustomFields
                // or includeLatestVersion if you want to retrieve custom fields and/or version information.
                var productInfo = licenseManager.GetProductDetails(false, false);

                Console.WriteLine( "------------- Product info -------------" );
                Console.WriteLine( "Product name:         " + productInfo.ProductName );
                Console.WriteLine( "Is trial allowed:     " + productInfo.IsTrialAllowed );
                Console.WriteLine( "Metadata:             " + productInfo.Metadata );
                if( productInfo.IsTrialAllowed && productInfo.TrialPeriod > 0 )
                {
                    string periodStr = "Trial period:         " + productInfo.TrialPeriod + " day";
                    if( productInfo.TrialPeriod > 1 )
                        periodStr += "s";
                    Console.WriteLine( periodStr );
                }

                SampleBase sample = null;
                string authMethod = "Authorization method: ";

                if( productInfo.AuthorizationType == AuthorizationMethod.KeyBased )
                {
                    sample = new KeyBasedSample();
                    authMethod += "Key-based";
                }
                else
                {
                    sample = new UserBasedSample();
                    authMethod += "User-based";
                }

                Console.WriteLine( authMethod );
                Console.WriteLine();

                // Run appropriate sample
                if( licenseManager.IsOnline() )
                    sample.RunOnline( licenseManager, deactivateAndRemove );
                else
                    sample.RunOffline( licenseManager, deactivateAndRemove );
            }
            catch( Exception ex )
            {
                SampleBase.PrintException( ex );
                SampleBase.PrintPause();
            }

            // Short YubiKey example
            if (false)
            {
                List<string> availableKeySerialNumbers = HardwareKeyOptions.ListAvailableKeys();
                if (availableKeySerialNumbers.Count == 0)
                {
                    Console.WriteLine("No YubiKey devices detected.");
                    return;
                }
                else
                {
                    foreach (string keySerialNumber in availableKeySerialNumbers)
                    {
                        Console.WriteLine($"Available key serial number: {keySerialNumber}");
                    }
                }

                string serialNumber = availableKeySerialNumbers.First();
                string pin = "123456";
                
                // Create your configuration object like you usually do.
                Configuration yubiKeyConfiguration = new Configuration();
                
                // Use ExtendedOptions to activate your hardware key
                HardwareKeyOptions options = new HardwareKeyOptions(serialNumber, pin);
                yubiKeyConfiguration.ExtendedOptions.ActivateHardwareKey(options);

                // Initialize your LicenseManager like you usually do.
                LicenseManager yubiKeyLicenseManager = new LicenseManager();
                yubiKeyLicenseManager.Initialize(yubiKeyConfiguration);

                // Activate your license
                string licenseKey = "your-license-key";
                LicenseID id = LicenseID.FromKey(licenseKey);
                ILicense license = yubiKeyLicenseManager.ActivateLicense(licenseKey);
            }
        }

        public static void TestFloatingServer( Configuration lsConfig )
        {
            var floatingClient = new FloatingClient();
            floatingClient.Initialize( lsConfig );

            Console.WriteLine( "Checking connection to Floating server..." );
            if( floatingClient.IsOnline( true ) )
            {
                Console.WriteLine( "Connection established." );
                Console.WriteLine();
            }

            string clientId; // can be anything, e.g. identifier of this user, machine id or app instance id
            var license = floatingClient.CurrentLicense();
            if( license == null )
                // for example let it be host name + process id
                clientId = lsConfig.Hostname +
                    System.Diagnostics.Process.GetCurrentProcess().Id;
            else
                clientId = license.FloatingClientId();

            Console.WriteLine( "Registering client: " + clientId );
            license = floatingClient.Register( clientId );
            Console.WriteLine( "Registration successful." );
            Console.WriteLine();

            SampleBase.PrintLicense( license );

            Console.WriteLine( "Unregistering client: " + clientId );
            if( floatingClient.Unregister( clientId ) )
                Console.WriteLine( "Unregistered successfully." );

            SampleBase.PrintPause();
        }
    }
}
