﻿using System;
using System.Linq;
using System.Threading;
using LicenseSpring;

namespace Sample.LicenseSpring
{
    class FloatingLicenseHelper
    {
        public static void RunLicenseWatchdog( ILicense license )
        {
            _license = license;
            _license.SetupLicenseWatchdog( HandleErrors );
        }

        private static void HandleErrors( LicenseSpringException ex )
        {
            if( ex == null )
            {
                Console.WriteLine( "\nLicense successfully checked: " + _license.LastCheckDate() );
            }
            else
            {
                Console.WriteLine( "\nLicense watchdog encountered error." );
                SampleBase.PrintException( ex );
                if( ex is NetworkException || ex is LicenseServerException )
                    _license.ResumeLicenseWatchdog(); // ignore not critical errors
                else if( ex is MaxFloatingReachedException )
                    Console.WriteLine( "Application cannot use this license at the moment because floating license limit reached." );
            }
        }

        private static ILicense _license;
    }

    abstract class SampleBase
    {
        public abstract void RunOnline( ILicenseManager licenseManager, bool deactivateAndRemove = false );

        public abstract void RunOffline( ILicenseManager licenseManager, bool deactivateAndRemove = false );

        public void UpdateAndCheckLicense( ILicense license )
        {
            // Increase and update license consumption
            if( license.Type() == LicenseType.Consumption )
            {
                license.UpdateConsumption();
                license.SyncConsumption();
            }

            // Increase and update license features consumption
            LicenseFeature[] licenseFeatures = license.Features();
            foreach( LicenseFeature feature in licenseFeatures )
            {
                if( feature.FeatureType == LicenseFeature.Type.Consumption )
                    license.UpdateFeatureConsumption( feature.Code );
            }
            license.SyncFeatureConsumption(); // sync consumption of all features

            // Register floating license features
            foreach (ILicenseFeature feature in license.Features())
            {
                // RegisterFloatingFeature performs the same check internally so it's not mandatory.
                if (feature.IsFloating || feature.IsOfflineFloating)
                {
                    try
                    {
                        license.RegisterFloatingFeature(feature);
                        // Feature data is not refreshed (feature variable doesn't contain updated FloatingSlotsInUse value)
                        // so we need to get new license features by calling license.Features() again.
                        LicenseFeature updatedFeature = license.Features().First(e => e.Code == feature.Code);
                        uint slotsInUse = updatedFeature.FloatingSlotsInUse;
                        Console.WriteLine($"Floating feature {feature.Code} successfuly registered. Current devices in use: {slotsInUse}.");

                        // Some more information:
                        uint floatingTimeout = feature.FloatingTimeout;
                        DateTime endDateTime = updatedFeature.FloatingEndDateTime;
                        DateTime endDateTimeUtc = updatedFeature.FloatingEndDateTimeUtc;
                    }
                    catch (Exception ex)
                    {
                        PrintException(ex);
                    }
                }
            }

            // Release floating license features
            foreach (ILicenseFeature feature in license.Features())
            {
                // ReleaseFloatingFeature performs the same check internally so it's not mandatory.
                if(feature.IsFloating || feature.IsOfflineFloating)
                {
                    try
                    {
                        license.ReleaseFloatingFeature(feature);
                        // Feature data is not refreshed (feature variable doesn't contain updated FloatingSlotsInUse value)
                        // so we need to get new license features by calling license.Features() again.
                        LicenseFeature updatedFeature = license.Features().First(e => e.Code == feature.Code);
                        uint slotsInUse = updatedFeature.FloatingSlotsInUse;
                        Console.WriteLine($"Floating feature {feature.Code} successfuly released. Current devices in use: {slotsInUse}.");

                        // Some more information:
                        uint floatingTimeout = feature.FloatingTimeout;
                        DateTime endDateTime = updatedFeature.FloatingEndDateTime;
                        DateTime endDateTimeUtc = updatedFeature.FloatingEndDateTimeUtc;
                    }
                    catch (Exception ex)
                    {
                        PrintException(ex);
                    }
                }
            }

            // Example of sending custom data to the LS platform (see device variables on the platform)
            Console.WriteLine( "Sending custom data to the LicenseSpring platform..." );
            license.AddDeviceVariable( new DeviceVariable( "CPU_Cores", Environment.ProcessorCount.ToString() ) );
            license.SendDeviceVariables();
            Console.WriteLine( "Operation completed successfully\n" );

            // You can omit above calls which update consumption, it will be updated automatically during license check
            // Sync license with the platform.
            if( license.IsFloating() )
            {
                if( license.IsBorrowed() )
                    return; // There is no need to register borrowed license online.
                if( license.CanBorrow() )
                {
                    Console.WriteLine( "Attempting to borrow license..." );
                    license.Borrow();
                    Console.WriteLine( "Operation completed successfully\n" );
                }
                else
                {
                    FloatingLicenseHelper.RunLicenseWatchdog( license );
                    Console.WriteLine( "Background thread for automatic license checks started...\n" );
                    Thread.Sleep( 1500 ); // Let background thread update the license
                }
            }
            else
            {
                Console.WriteLine( "Checking license online..." );

                // set to true if you want to see expired features in your license data. Set to false if you only need active features.
                bool includeExpiredFeatures = false;
                license.Check(null, includeExpiredFeatures); // throws exceptions in case of errors
                Console.WriteLine( "License successfully checked\n" );
                if( license.IsGracePeriodStarted() )
                {
                    Console.WriteLine( "Grace period started!");
                    Console.WriteLine( "{0} hours till the end of grace period.", license.GracePeriodHoursRemaining() );
                    Console.WriteLine( "Grace period end date time: {0}.", license.GracePeriodEndDateTime() );
                }
            }
        }

        public void CheckLicenseLocal( ILicenseManager manager, ref ILicense license )
        {
            Console.WriteLine( "License successfully loaded, performing local check of the license..." );
            // it's highly recommended to perform a localCheck (offline check) on every startup
            // to be ensure that license file wasn't copied from another PC and license in a valid state
            try
            {
                license.LocalCheck(); // throws exceptions in case of errors, see documentation
            }
            catch( DeviceNotLicensedException ex )
            {
                // Below is an example on how to upgrade to new or other device id algorithm
                SampleBase.PrintException( ex );
                Console.WriteLine( "Trying to upgrade to newer device id algorithm..." );
                license = manager.RelinkLicense( DeviceIDAlgorithm.Gen3 );
                Console.WriteLine( "License successfully linked to new device id.\n" );
                return;
            }
            catch( FloatingTimeoutExpiredException ex )
            {
                SampleBase.PrintException( ex );
                Console.WriteLine( "Registration of this floating license has expired at: {0}",
                                   license.FloatingEndDateTime() );
                Console.WriteLine( "Trying to register floating license..." );
                license.Register();
                Console.WriteLine( "License successfully checked in.\n" );
                return;
            }
            Console.WriteLine( "Local validation successful\n" );
        }

        public void CleanUp( ILicense license, bool deactivateAndRemove )
        {
            if( !license.IsBorrowed() )
                license.Release();
            if( deactivateAndRemove )
            {
                if( license.Deactivate( true ) )
                    Console.WriteLine( "License deactivated successfully." );
                Console.WriteLine();
            }
            PrintPause();
        }

        public void CleanUpLocal( ILicenseManager manager, ILicense license, bool deactivateAndRemove )
        {
            if( deactivateAndRemove )
            {
                var filePath = license.DeactivateOffline();
                manager.ClearLocalStorage();
                Console.WriteLine( "To finish deactivation process please upload deactivation request file to the LicenseSpring portal." );
                Console.WriteLine( "File path: " + filePath );
                Console.WriteLine( "Offline licensing portal address: https://offline.licensespring.com" );
                Console.WriteLine();
            }
            PrintPause();
        }

        public void CreateOfflineActivationRequest( ILicenseManager manager, LicenseID licenseId )
        {
            Console.WriteLine( "Creating offline activation request file..." );
            var filePath = manager.GetOfflineActivationFile( licenseId );
            Console.WriteLine( "File created: " + filePath );
            Console.WriteLine( "Please upload that request file to LicenseSpring offline activation portal to get response file." );
            Console.WriteLine( "Offline activation portal address: https://offline.licensespring.com" );
            PrintPause();
        }

        public void UpdateOfflineLicense( ILicense license )
        {
            // Here is an example how to update offline license.
            // You can download refresh file from LicenseSpring platform, see Devices section of particular license.
            // Choose active device for which you would like to update the license and press "Download license refresh file" button.
            // You may also create refresh file using Management API.

            // Assign license refresh file path below
            string updateFilePath = "";
            // updateFilePath = Environment.GetFolderPath( Environment.SpecialFolder.Desktop ) + "\\license_refresh.lic";
            if( license.UpdateOffline( updateFilePath ) )
                Console.WriteLine( "License updated\n" );
        }

        public static void PrintLicense( ILicense license )
        {
            if( license == null )
                return;

            Console.WriteLine( "------------- License info -------------" );

            var owner = license.Owner();
            if( owner != null )
            {
                string ownerInfo = "";
                Action<string> formatInfo = dataField =>
                {
                    if( !string.IsNullOrEmpty( ownerInfo ) )
                        ownerInfo += " ";
                    if( !string.IsNullOrEmpty( dataField ) )
                        ownerInfo += dataField;
                };
                formatInfo( owner.FirstName );
                formatInfo( owner.LastName );
                formatInfo( owner.Email );
                formatInfo( owner.Company );
                formatInfo( owner.Metadata );

                if( !string.IsNullOrEmpty( ownerInfo ) )
                    Console.WriteLine( "Licensed to: {0}", ownerInfo );
            }

            if( !string.IsNullOrEmpty( license.Key() ) )
                Console.WriteLine( "Key = {0}", license.Key() );

            if( !string.IsNullOrEmpty( license.User() ) )
                Console.WriteLine( "User = {0}", license.User() );

            Console.WriteLine( "Type = {0}", license.Type() );
            Console.WriteLine( "Status = {0}", license.Status().ToString() );
            Console.WriteLine( "IsTrial = {0}", license.IsTrial() );
            Console.WriteLine( "IsOfflineActivated = {0}", license.IsOfflineActivated() );

            Console.WriteLine( "Transfer count = {0}", license.TransferCount() );
            if( license.IsDeviceTransferAllowed() )
            {
                if( license.IsDeviceTransferLimited() )
                    Console.WriteLine( $"Device transfers limit = {license.TransferLimit()}" );
                else
                    Console.WriteLine( "Transferring license between devices is allowed and unlimited." );
            }
            else
                Console.WriteLine( "Device transfers not allowed!" );

            Console.WriteLine( "Times activated = {0}", license.TimesActivated() );
            Console.WriteLine( "Max activations = {0}", license.MaxActivations() );
            Console.WriteLine( "Start Date = {0}", license.StartDate() );
            Console.WriteLine( "Validity Period = {0}", license.ValidityPeriod() );
            Console.WriteLine( "Validity Period UTC = {0}", license.ValidityPeriodUTC() );
            Console.WriteLine( "Days remaining until license expires = {0}", license.DaysRemaining() );
            Console.WriteLine( "Metadata = {0}", license.Metadata() );
            if( license.Type() == LicenseType.Subscription )
            {
                if( license.IsSubscriptionGracePeriodStarted() )
                {
                    var remainingTime = license.ValidityWithGracePeriod() - DateTime.Now;
                    Console.WriteLine( "Subscription grace period started, remaining time: {0}", remainingTime );
                }
                else
                    Console.WriteLine( "Subscription grace period (hours) = {0}", license.SubscriptionGracePeriod() );
            }
            Console.WriteLine( "Maintenance period = {0}", license.MaintenancePeriod() );
            Console.WriteLine( "Maintenance period UTC = {0}", license.MaintenancePeriodUTC() );
            Console.WriteLine( "Maintenance days remaining = {0}", license.MaintenanceDaysRemaining() );
            Console.WriteLine( "Is maintenance period expired = {0}", license.IsMaintenancePeriodExpired() );
            Console.WriteLine( "Last online check date = {0}", license.LastCheckDate() );
            Console.WriteLine( "Days passed since last online check = {0}", license.DaysPassedSinceLastCheck() );

            var productFeatures = license.Features();
            if( productFeatures != null && productFeatures.Count() > 0 )
            {
                Console.WriteLine( "Product features available for this license:" );
                foreach( LicenseFeature feature in productFeatures )
                    Console.WriteLine( feature.ToString() );
            }

            var dataFields = license.CustomFields();
            if( dataFields != null && dataFields.Count() > 0 )
            {
                Console.WriteLine( "Custom data fields available for this license:" );
                foreach( CustomField field in dataFields )
                    Console.WriteLine( "Data field - Name: {0}, Value: {1}", field.Name, field.Value );
            }

            var variables = license.GetDeviceVariables( true );
            if( variables != null && variables.Count() > 0 )
            {
                Console.WriteLine( "Device variables available for this license:" );
                foreach( DeviceVariable variable in variables )
                    Console.WriteLine( "Device variable - Name: {0}, Value: {1}", variable.Name, variable.Value );
            }

            if( license.Type() == LicenseType.Consumption )
            {
                Console.WriteLine( "Total consumption = {0}", license.TotalConsumption() );
                if( license.IsUnlimitedConsumptionAllowed() )
                    Console.WriteLine( "Max consumption = Unlimited" );
                else
                    Console.WriteLine( "Max consumption = {0}", license.MaxConsumption() );
                Console.WriteLine( "Is consumption overage allowed = {0}", license.IsConsumptionOverageAllowed() );
                if( license.IsConsumptionOverageAllowed() )
                    Console.WriteLine( "Max consumption overage = {0}", license.MaxConsumptionOverage() );
            }

            if( license.IsFloating() )
            {
                Console.WriteLine( "This license is floating" );
                Console.WriteLine( "Timeout = {0} min", license.FloatingTimeout() );
                Console.WriteLine( "Current floating slots count = {0}", license.FloatingSlotsInUse() );
                Console.WriteLine( "Overall floating slots count = {0}", license.FloatingSlotsCount() );
                if( !string.IsNullOrEmpty( license.FloatingClientId() ) )
                    Console.WriteLine( "Floating user (client) id = {0}", license.FloatingClientId() );
                if( license.IsBorrowed() )
                    Console.WriteLine( "The license is borrowed until {0}", license.FloatingEndDateTime() );
                else
                {
                    if( license.CanBorrow() )
                        Console.WriteLine( "License can be borrowed for {0} hours max", license.MaxBorrowTime() );
                    else
                        Console.WriteLine( "License borrowing is not allowed" );
                }
            }

            Console.WriteLine();
        }

        public static void GetAndDisplayVersionsInfo( ILicenseManager manager )
        {
            var license = manager.CurrentLicense();
            var allVersions = manager.GetAllVersions( license.Id() ).ToList();
            if( allVersions.Any() )
            {
                var installFile = manager.GetInstallationFile( license.Id(),
                    new InstallationFileOptions( allVersions.Last() ) );
                PrintInstallFile( installFile );
            }
        }

        public static void PrintInstallFile( InstallationFile installFile )
        {
            if( installFile == null )
                return;

            Console.WriteLine( "------------- Update info -------------" );
            Console.WriteLine( "Install file for app version: {0}", installFile.Version );
            Console.WriteLine( "URL for downloading: {0}", installFile.Url );
            Console.WriteLine( "Md5 hash: {0}", installFile.Md5Hash );
            Console.WriteLine( "Release date: {0}", installFile.ReleaseDate.ToString( "dd.MM.yyyy" ) );
            if( !string.IsNullOrEmpty( installFile.RequiredVersion ) )
                Console.WriteLine( "Required app version: {0}", installFile.RequiredVersion );
            Console.WriteLine( "EULA link: {0}", installFile.EulaLink );
            Console.WriteLine( "Release notes link: {0}", installFile.ReleaseNotesLink );
            Console.WriteLine( "Environment: {0}", installFile.Environment );
            if( installFile.Size > 0 )
                Console.WriteLine( "Size: {0} bytes", installFile.Size );
            if( !string.IsNullOrEmpty( installFile.Channel ) )
                Console.WriteLine( "Channel: {0}", installFile.Channel );
            Console.WriteLine();
        }

        public static void PrintException( Exception ex )
        {
            Console.WriteLine( "Exception encountered {0}: {1}", ex.GetType().FullName, ex.Message );
            Exception innerEx = ex.InnerException;
            while( innerEx != null )
            {
                Console.WriteLine( "Inner exception {0}: {1}", innerEx.GetType().FullName, innerEx.Message );
                innerEx = innerEx.InnerException;
            }
            Console.WriteLine();
        }

        public static void PrintPause()
        {
            Console.WriteLine( "\nPlease press any key to continue..." );
            Console.ReadKey();
        }
    }
}
