﻿Imports System.Linq
Imports LicenseSpring

Public MustInherit Class SampleBase

    Public MustOverride Sub RunOnline(ByRef LicenseManager As ILicenseManager, Optional deactivateAndRemove As Boolean = False)

    Public MustOverride Sub RunOffline(ByRef licenseManager As ILicenseManager, Optional deactivateAndRemove As Boolean = False)

    Public Sub UpdateAndCheckLicense(ByRef license As ILicense)
        'update consumption
        If (license.Type() = LicenseType.Consumption) Then
            license.UpdateConsumption()
            license.SyncConsumption()
        End If

        ' sync license with the platform
        Console.WriteLine("Checking license online...")
        license.Check()  'throws exceptions In Case Of errors
        Console.WriteLine("License successfully checked\n")

        ' Increase consumption features
        For Each feature As LicenseFeature In license.Features()
            If (feature.FeatureType = LicenseFeature.Type.Consumption) Then
                license.UpdateFeatureConsumption(feature.Code)
            End If
        Next

        ' Example of sending custom data to the LS platform (see device variables on the platform)
        Console.WriteLine("Sending custom data to the LicenseSpring platform...")
        license.AddDeviceVariable(New DeviceVariable("CPU_Cores", Environment.ProcessorCount.ToString()))
        license.SendDeviceVariables()
        Console.WriteLine("Operation completed successfully\n")
    End Sub

    Public Sub CheckLicenseLocal(ByRef license As ILicense)
        Console.WriteLine("License successfully loaded, performing local check of the license...")
        ' it's highly recommended to perform a localCheck (offline check) on every startup
        ' to be ensure that license file wasn't copied from another PC and license in a valid state
        license.LocalCheck() ' throws exceptions In Case Of errors, see documentation
        Console.WriteLine("Local validation successful\n")
    End Sub

    Public Sub CleanUp(ByRef manager As ILicenseManager, ByRef license As ILicense)
        license.Deactivate()
        Console.WriteLine("License deactivated successfully.")
        manager.ClearLocalStorage()
        Console.WriteLine()
    End Sub

    Public Sub CleanUpLocal(ByRef manager As ILicenseManager, ByRef license As ILicense)
        Dim filePath = license.DeactivateOffline()
        manager.ClearLocalStorage()
        Console.WriteLine("To finish deactivation process please upload deactivation request file to the LicenseSpring portal.")
        Console.WriteLine("File path: " + filePath)
        Console.WriteLine()
    End Sub

    Public Sub PrintLicense(ByRef license As ILicense)
        If (license Is Nothing) Then
            Return
        End If

        Console.WriteLine("------------- License info -------------")

        If (Not String.IsNullOrEmpty(license.Key())) Then
            Console.WriteLine("Key = {0}", license.Key())
        End If

        If (Not String.IsNullOrEmpty(license.User())) Then
            Console.WriteLine("User = {0}", license.User())
        End If

        Console.WriteLine("Type = {0}", license.Type())
        Console.WriteLine("Status = {0}", license.Status().ToString())
        Console.WriteLine("IsTrial = {0}", license.IsTrial())
        Console.WriteLine("IsOfflineActivated = {0}", license.IsOfflineActivated())

        Console.WriteLine("Transfer count = {0}", license.TransferCount())
        If (license.IsDeviceTransferAllowed()) Then
            If (license.IsDeviceTransferLimited()) Then
                Console.WriteLine($"Device transfers limit = {license.TransferLimit()}")
            Else
                Console.WriteLine("Transferring license between devices is allowed and unlimited.")
            End If
        Else
            Console.WriteLine("Device transfers not allowed!")
        End If

        Console.WriteLine("Times activated = {0}", license.TimesActivated())
        Console.WriteLine("Max activations = {0}", license.MaxActivations())
        Console.WriteLine("Validity Period = {0}", license.ValidityPeriod())
        Console.WriteLine("Validity Period UTC = {0}", license.ValidityPeriodUTC())
        Console.WriteLine("Maintenance period = {0}", license.MaintenancePeriod())
        Console.WriteLine("Maintenance period UTC = {0}", license.MaintenancePeriodUTC())
        Console.WriteLine("Last online check date = {0}", license.LastCheckDate())
        Console.WriteLine("Days passed since last online check = {0}", license.DaysPassedSinceLastCheck())

        If (license.Type() = LicenseType.Subscription) Then
            Console.WriteLine("Subscription grace period (hours) = {0}", license.SubscriptionGracePeriod())
        End If

        Dim productFeatures = license.Features()
        If (productFeatures IsNot Nothing And productFeatures.Count() > 0) Then
            Console.WriteLine("Product features available for this license:")
            For Each feature In productFeatures
                Console.WriteLine(feature.ToString())
            Next
        End If

        Dim dataFields = license.CustomFields()
        If (dataFields IsNot Nothing And dataFields.Count() > 0) Then
            Console.WriteLine("Custom data fields available for this license:")
            For Each field In dataFields
                Console.WriteLine("Data field - Name: {0}, Value: {1}", field.Name, field.Value)
            Next
        End If

        If (license.Type() = LicenseType.Consumption) Then
            Console.WriteLine("Total consumptions = {0}", license.TotalConsumption())
            If (license.IsUnlimitedConsumptionAllowed()) Then
                Console.WriteLine("Max consumption = Unlimited")
            Else
                Console.WriteLine("Max consumptions = {0}", license.MaxConsumption())
            End If
            Console.WriteLine("Is consumption overage allowed = {0}", license.IsConsumptionOverageAllowed())
            If (license.IsConsumptionOverageAllowed()) Then
                Console.WriteLine("Max consumption overage = {0}", license.MaxConsumptionOverage())
            End If
        End If

        If (license.IsFloating()) Then
            Console.WriteLine("This license is floating")
            Console.WriteLine("Timeout = {0} min", license.FloatingTimeout())
            Console.WriteLine("Current floating slots count = {0}", license.FloatingSlotsInUse())
            Console.WriteLine("Overall floating slots count = {0}", license.FloatingSlotsCount())
        End If

        Console.WriteLine()
    End Sub

    Public Sub GetAndDisplayVersionsInfo(manager As ILicenseManager)
        If (manager Is Nothing) Then
            Return
        End If

        Dim license = manager.CurrentLicense
        Dim allVersions = manager.GetAllVersions(license.Id()).ToList()
        If (allVersions.Any()) Then
            Dim installFile = manager.GetInstallationFile(license.Id(), New InstallationFileOptions(allVersions.Last()))
            PrintInstallFile(installFile)
        End If
    End Sub

    Public Sub PrintInstallFile(installFile As InstallationFile)
        If (installFile Is Nothing) Then
            Return
        End If

        Console.WriteLine("------------- Update info -------------")
        Console.WriteLine("Install file for app version: {0}", installFile.Version)
        Console.WriteLine("URL for downloading: {0}", installFile.Url)
        Console.WriteLine("Md5 hash: {0}", installFile.Md5Hash)
        Console.WriteLine("Release date: {0}", installFile.ReleaseDate.ToString("dd.MM.yyyy"))
        Console.WriteLine("Required app version: {0}", installFile.RequiredVersion)
        Console.WriteLine("EULA link: {0}", installFile.EulaLink)
        Console.WriteLine("Release notes link: {0}", installFile.ReleaseNotesLink)
        Console.WriteLine("Environment: {0}", installFile.Environment)
        Console.WriteLine("Size: {0} bytes", installFile.Size)
        Console.WriteLine("Channel: {0}", installFile.Channel)
        Console.WriteLine()
    End Sub

End Class
