﻿using Avalonia.Controls;
using LicenseSpring;
using System.Text.Json;
using ReactiveUI;
using System;
using System.Collections.Generic;
using System.IO;
using System.Reactive;
using System.Collections.ObjectModel;

namespace LSCrossPlatformSample
{
    class InfoWindowViewModel : ReactiveObject
    {
        private Window _view;

        public Window view
        {
            set
            {
                _view = value;
            }
            get
            {
                return _view;
            }
        }


        public ObservableCollection<InstallationFile> InstallationFiles { get; set; }
        public ObservableCollection<LicenseFeature> LicenseFeatures { get; set; }
        public ObservableCollection<CustomField> CustomFields { get; set; }
        public ReactiveCommand<Unit, Unit> OpenLicenseFolder { get; }
        public ReactiveCommand<Unit, Unit> ClearLicenseFolder { get; }
        public ReactiveCommand<Unit, Unit> UpdateConsumption { get; }
        public ReactiveCommand<Unit, Unit> SyncConsumption { get; }
        public ReactiveCommand<Unit, Unit> CheckLicense { get; }
        public ReactiveCommand<Unit, Unit> DeactivateLicense { get; }
        public ReactiveCommand<Unit, Unit> LoadAppversionList { get; }
        public ReactiveCommand<Unit, Unit> CheckUpdates { get; }
        public ReactiveCommand<Unit, Unit> Settings { get; }
        public ReactiveCommand<Unit, Unit> ProductDetails { get; }
        public ReactiveCommand<LicenseFeature, Unit> SyncFeatureConsumption { get; }
        public ReactiveCommand<LicenseFeature, Unit> UpdateFeatureConsumption { get; }

        private string licensePath;
        private string key;
        private string type;
        private string status;
        private bool isTrial;
        private string validityPeriod;
        private int daysRemaining;
        private string maintenancePeriod;
        private DateTime lastChecked;
        private int totalCount;
        private int maxCount;
        private string appVersion;
        private int updateConsumptionValue;
        private int updateFeatureConsumptionValue;
        private string user;
        private bool isOfflineActivated;
        private bool isValid;
        private bool isFloating;
        private uint floatingTimeout;


        public string User
        {
            get => user;
            set => this.RaiseAndSetIfChanged(ref user, value);
        }
        public bool IsOfflineActivated
        {
            get => isOfflineActivated;
            set => this.RaiseAndSetIfChanged(ref isOfflineActivated, value);
        }
        public bool IsValid
        {
            get => isValid;
            set => this.RaiseAndSetIfChanged(ref isValid, value);
        }
        public bool IsFloating
        {
            get => isFloating;
            set => this.RaiseAndSetIfChanged(ref isFloating, value);
        }
        public uint FloatingTimeout
        {
            get => floatingTimeout;
            set => this.RaiseAndSetIfChanged(ref floatingTimeout, value);
        }
        public int UpdateConsumptionValue
        {
            get => updateConsumptionValue;
            set => this.RaiseAndSetIfChanged(ref updateConsumptionValue, value);
        }
        public int UpdateFeatureConsumptionValue
        {
            get => updateFeatureConsumptionValue;
            set => this.RaiseAndSetIfChanged(ref updateFeatureConsumptionValue, value);
        }
        public string LicensePath
        {
            get => licensePath;
            set => this.RaiseAndSetIfChanged(ref licensePath, value);
        }
        public string Key
        {
            get => key;
            set => this.RaiseAndSetIfChanged(ref key, value);
        }
        public string Type
        {
            get => type;
            set => this.RaiseAndSetIfChanged(ref type, value);
        }
        public string Status
        {
            get => status;
            set => this.RaiseAndSetIfChanged(ref status, value);
        }
        public bool IsTrial
        {
            get => isTrial;
            set => this.RaiseAndSetIfChanged(ref isTrial, value);
        }
        public string ValidityPeriod
        {
            get => validityPeriod;
            set => this.RaiseAndSetIfChanged(ref validityPeriod, value);
        }
        public int DaysRemaining
        {
            get => daysRemaining;
            set => this.RaiseAndSetIfChanged(ref daysRemaining, value);
        }
        public string MaintenancePeriod
        {
            get => maintenancePeriod;
            set => this.RaiseAndSetIfChanged(ref maintenancePeriod, value);
        }
        public DateTime LastChecked
        {
            get => lastChecked;
            set => this.RaiseAndSetIfChanged(ref lastChecked, value);
        }
        public int TotalCount
        {
            get => totalCount;
            set => this.RaiseAndSetIfChanged(ref totalCount, value);
        }
        public int MaxCount
        {
            get => maxCount;
            set => this.RaiseAndSetIfChanged(ref maxCount, value);
        }
        public string AppVersion
        {
            get => appVersion;
            set => this.RaiseAndSetIfChanged(ref appVersion, value);
        }
        private LicenseSpring.Configuration configuration = null;
        public bool isDeactivated = false;
        public InfoWindowViewModel(LicenseSpring.Configuration configuration)
        {
            this.configuration = configuration;

            InstallationFiles = new ObservableCollection<InstallationFile>();
            LicenseFeatures = new ObservableCollection<LicenseFeature>();
            CustomFields = new ObservableCollection<CustomField>();

            UpdateLicenseInfo();
            LoadFeatureList();
            OpenLicenseFolder = ReactiveCommand.Create(OnOpenLicenseFolder);
            ClearLicenseFolder = ReactiveCommand.Create(OnClearLicenseFolder);
            UpdateConsumption = ReactiveCommand.Create(OnUpdateConsumption);
            SyncConsumption = ReactiveCommand.Create(OnSyncConsumption);
            CheckLicense = ReactiveCommand.Create(OnCheckLicense);
            DeactivateLicense = ReactiveCommand.Create(OnDeactivateLicense);
            LoadAppversionList = ReactiveCommand.Create(OnLoadAppversionList);
            CheckUpdates = ReactiveCommand.Create(OnCheckUpdates);
            Settings = ReactiveCommand.Create(OnSettings);
            SyncFeatureConsumption = ReactiveCommand.Create<LicenseFeature>(OnSyncFeatureConsumption);
            UpdateFeatureConsumption = ReactiveCommand.Create<LicenseFeature>(OnUpdateFeatureConsumption);
            ProductDetails = ReactiveCommand.Create(OnProductDetails);
        }
        public void InitializeLicenseSpring()
        {
            try
            {
                configuration = LSConfig.CreateConfiguration();
                ConfigData data = readConfigData();
                if (data != null)
                {
                    ExtendedOptions options = configuration.ExtendedOptions;
                    options.CollectNetworkInfo = data.CollectNetworkInfo;
                    options.EnableLogging = data.EnableLogging;
                    if (data.ProxyUri != null && data.ProxyUri.Length > 0)
                    {
                        if (data.ProxyUser != null && data.ProxyUser.Length > 0 && data.ProxyPassword != null && data.ProxyPassword.Length > 0)
                            options.Proxy = new ProxySettings(data.ProxyUser, data.ProxyPassword, data.ProxyUri);
                        else
                            options.Proxy = new ProxySettings(data.ProxyUri, data.UseDefaultCredentials);
                    }
                    else if (data.ProxyUser != null && data.ProxyUser.Length > 0 && data.ProxyPassword != null && data.ProxyPassword.Length > 0)
                        options.Proxy = new ProxySettings(data.ProxyUser, data.ProxyPassword);
                    configuration.ExtendedOptions = options;
                }
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                licenseManager.Initialize(configuration);
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }
        private async void OnSettings()
        {
            try
            {
                var window = new ConfigWindow()
                {
                    DataContext = new ConfigWindowViewModel()
                };
                await window.ShowDialog(view);
                InitializeLicenseSpring();
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        private void OnSyncFeatureConsumption(LicenseFeature feature)
        {
            try
            {
                if (feature != null)
                {
                    var licenseManager = LicenseManager.GetInstance();
                    var license = licenseManager.CurrentLicense();

                    license.SyncFeatureConsumption(feature.Code);
                    LoadFeatureList();
                }
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }
        private void OnUpdateFeatureConsumption(LicenseFeature feature)
        {
            try
            {
                if (feature != null)
                {
                    var licenseManager = LicenseManager.GetInstance();
                    var license = licenseManager.CurrentLicense();
                    license.UpdateFeatureConsumption(feature.Code, UpdateFeatureConsumptionValue);
                    LoadFeatureList();
                }
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }
        private void UpdateLicenseInfo()
        {
            try
            {
                var licenseManager = LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                Key = license.Key();
                Type = license.Type().ToString();
                Status = license.Status().ToString();
                IsTrial = license.IsTrial();
                ValidityPeriod = license.ValidityPeriod().ToString("dd/MM/yyyy");
                DaysRemaining = license.DaysRemaining();
                MaintenancePeriod = license.MaintenancePeriod().ToString("dd/MM/yyyy");
                LastChecked = license.LastCheckDate();
                TotalCount = license.TotalConsumption();
                MaxCount = license.MaxConsumption();
                User = license.User();
                IsOfflineActivated = license.IsOfflineActivated();
                IsValid = license.IsValid();
                IsFloating = license.IsFloating();
                FloatingTimeout = license.FloatingTimeout();
                AppVersion = (null != configuration) ? configuration.AppVersion : "";
                LicensePath = Path.GetFullPath(Path.Combine(licenseManager.DataLocation, licenseManager.LicenseFileName));
                if (CustomFields != null) CustomFields.Clear();
                var fields = license.CustomFields();
                foreach (var field in fields)
                {
                    CustomFields.Add(field);
                }
                LoadFeatureList();
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }
        void OnOpenLicenseFolder()
        {
            var warningMessage = "Couldn't open license file directory!";
            try
            {
                if (null == configuration)
                {
                    MsgBox.Show(_view, warningMessage, "Error", MsgBox.MessageBoxButtons.Ok);
                    return;
                }
                var path = Path.GetFullPath( LicenseManager.GetInstance().DataLocation );
                if (0 == path.Length)
                {
                    MsgBox.Show(_view, warningMessage, "Error", MsgBox.MessageBoxButtons.Ok);
                    return;
                }
                var dir = new DirectoryInfo( path );
                if (!dir.Exists)
                {
                    MsgBox.Show(_view, warningMessage, "Error", MsgBox.MessageBoxButtons.Ok);
                    return;
                }
                var psi = new System.Diagnostics.ProcessStartInfo() { FileName = dir.FullName, UseShellExecute = true };
                System.Diagnostics.Process.Start(psi);
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }
        void OnClearLicenseFolder()
        {
            try
            {
                var licenseManager = LicenseManager.GetInstance();
                licenseManager.ClearLocalStorage();
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }
        void OnUpdateConsumption()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                license.UpdateConsumption(updateConsumptionValue);
                TotalCount = license.TotalConsumption();
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }
        void OnSyncConsumption()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                license.SyncConsumption();
                TotalCount = license.TotalConsumption();
                MaxCount = license.MaxConsumption();
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }
        void OnCheckLicense()
        {
            try
            {
                var license = LicenseManager.GetInstance().CurrentLicense();
                if (null == license)
                {
                    MsgBox.Show(_view, "There is no license file!", "Error", MsgBox.MessageBoxButtons.Ok);
                    return;
                }
                license.Check();
                MsgBox.Show(_view, "License successfully checked!", "Success", MsgBox.MessageBoxButtons.Ok);
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
            UpdateLicenseInfo();
        }
        void OnDeactivateLicense()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                if (license != null) license.Deactivate();
                isDeactivated = true;
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
            _view.Close();
        }
        void OnLoadAppversionList()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                var versions = licenseManager.GetAllVersions(license.Id());
                InstallationFiles.Clear();
                foreach (var version in versions)
                {
                    var installationFile = licenseManager.GetInstallationFile(license.Id(),
                        new InstallationFileOptions(version));
                    InstallationFiles.Add(installationFile);
                }
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }
        void LoadFeatureList()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();

                var features = license.Features();
                LicenseFeatures.Clear();
                foreach (var feature in features)
                {
                    LicenseFeatures.Add(feature);
                }
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        void OnCheckUpdates()
        {
            if (null == configuration)
            {
                MsgBox.Show(_view, "Couldn't check current app version.", "Error", MsgBox.MessageBoxButtons.Ok);
                return;
            }
            try
            {
                var licenseManager = LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                var installationFile = licenseManager.GetInstallationFile(license.Id());
                if (string.IsNullOrEmpty(installationFile.Version))
                {
                    MsgBox.Show(_view, "Couldn't get version from the server!", "Error", MsgBox.MessageBoxButtons.Ok);
                    return;
                }
                Version currentVer = new Version(configuration.AppVersion);
                Version version = new Version(installationFile.Version);
                if (currentVer >= version)
                    MsgBox.Show(_view, "The application is up to date.", "Success", MsgBox.MessageBoxButtons.Ok);
                else
                    MsgBox.Show(_view, "There is a new version of the app available!", "Success", MsgBox.MessageBoxButtons.Ok);
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        async void OnProductDetails()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var details = licenseManager.GetProductDetails();
                var window = new ProductDetailsWindow()
                {
                    DataContext = new ProductDetailsWindowViewModel(details)
                };
                await window.ShowDialog(view);
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        private void HandleException(Exception ex)
        {
            if (ex is LicenseSpringException)
                MsgBox.Show(_view, "LicenseSpring exception occurred\n" + ex.Message, "Error", MsgBox.MessageBoxButtons.Ok);
            else
                MsgBox.Show(_view, "Exception occured\n" + ex.Message, "Error", MsgBox.MessageBoxButtons.Ok);
        }

        private ConfigData readConfigData()
        {
            ConfigData data;
            var licenseFolderPath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
            licenseFolderPath = Path.Combine(licenseFolderPath, "LicenseSpring");
            string fileName = Path.Combine(licenseFolderPath, "Settings.json");
            if (!File.Exists(fileName))
            {
                return null;
            }
            using (StreamReader r = new StreamReader(fileName))
            {
                string json = r.ReadToEnd();
                data = JsonSerializer.Deserialize<ConfigData>(json);
            }
            return data;
        }

    }
}
