﻿using Avalonia.Controls;
using LicenseSpring;
using System;
using System.Reactive;
using ReactiveUI;
using System.IO;
using System.Text.Json;
using System.Threading.Tasks;
using System.Diagnostics;

namespace LSCrossPlatformSample
{
    public class MainWindowViewModel : ReactiveObject
    {
        private Window _view;

        public Window view
        {
            set
            {
                _view = value;
            }
            get
            {
                return _view;
            }
        }
        public MainWindowViewModel()
        {
            ActivateKeyBased = ReactiveCommand.Create(OnActivateKeyBased);
            ActivateUserBased = ReactiveCommand.Create(OnActivateUserBased);
            GetTrialLicense = ReactiveCommand.Create(OnGetTrialLicense);
            Settings = ReactiveCommand.Create(OnSettings);
            ActivateKeyBasedOffline = ReactiveCommand.Create(OnActivateKeyBasedOffline);
            ActivateUserBasedOffline = ReactiveCommand.Create(OnActivateUserBasedOffline);
            ActivateOffline = ReactiveCommand.Create(OnActivateOffline);
            OpenActivationURL = ReactiveCommand.Create(OnOpenActivationURL);
            ProductDetails = ReactiveCommand.Create(OnProductDetails);
            IsKeyBased = true;
            IsUserBased = true;
        }

        public void onShowAsync()
        {
            view.Opened += (s, e) => InitializeLicenseSpring();
            view.Opened += (s, e) => TryToStartInfoScreen();
        }

        private string key;
        private string user;
        private string password;
        private string path;
        private bool isKeyBased;
        private bool isUserBased;
        public string Key
        {
            get => key;
            set => this.RaiseAndSetIfChanged(ref key, value);
        }
        public string User
        {
            get => user;
            set => this.RaiseAndSetIfChanged(ref user, value);
        }
        public string Password
        {
            get => password;
            set => this.RaiseAndSetIfChanged(ref password, value);
        }
        public string LicensePath
        {
            get => path;
            set => this.RaiseAndSetIfChanged(ref path, value);
        }
        public bool IsKeyBased
        {
            get => isKeyBased;
            set => this.RaiseAndSetIfChanged(ref isKeyBased, value);
        }
        public bool IsUserBased
        {
            get => isUserBased;
            set => this.RaiseAndSetIfChanged(ref isUserBased, value);
        }

        public ReactiveCommand<Unit, Unit> ActivateKeyBased { get; }
        public ReactiveCommand<Unit, Unit> ActivateUserBased { get; }
        public ReactiveCommand<Unit, Unit> ActivateTrial { get; }
        public ReactiveCommand<Unit, Unit> ActivateKeyBasedOffline { get; }
        public ReactiveCommand<Unit, Unit> ActivateUserBasedOffline { get; }
        public ReactiveCommand<Unit, Unit> ActivateTrialOffline { get; }
        public ReactiveCommand<Unit, Unit> ActivateOffline { get; }
        public ReactiveCommand<Unit, Unit> GetTrialLicense { get; }
        public ReactiveCommand<Unit, Unit> Settings { get; }
        public ReactiveCommand<Unit, Unit> ProductDetails { get; }
        public ReactiveCommand<Unit, Unit> OpenActivationURL { get; }

        LicenseSpring.Configuration configuration = null;
        private void TryToStartInfoScreen()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if (licenseManager == null)
                    return;
                var license = licenseManager.CurrentLicense();
                if (license == null)
                    return;
                if (license.IsValid())
                {
                    license.LocalCheck();
                    ShowInfoScreen();
                }
            }
            catch( Exception ) { }
        }
        private async void ShowInfoScreen()
        {
            view.Hide();
            var window = new InfoWindow()
            {
                DataContext = new InfoWindowViewModel(configuration)
            };
            await window.ShowDialog(view);
            if (((InfoWindowViewModel)window.DataContext).isDeactivated)
                view.Show();
            else
                view.Close();
        }

        public void InitializeLicenseSpring()
        {
            try
            {
                configuration = LSConfig.CreateConfiguration();
                configuration.AppName = System.Reflection.Assembly.GetExecutingAssembly().GetName().Name.ToString();
                configuration.AppVersion = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString();
                ConfigData data = readConfigData();
                if (data != null)
                {
                    ExtendedOptions options = configuration.ExtendedOptions;
                    options.CollectNetworkInfo = data.CollectNetworkInfo;
                    options.EnableLogging = data.EnableLogging;
                    if (data.ProxyUri != null && data.ProxyUri.Length > 0)
                    {
                        if (data.ProxyUser != null && data.ProxyUser.Length > 0 && data.ProxyPassword != null && data.ProxyPassword.Length > 0)
                            options.Proxy = new ProxySettings(data.ProxyUser, data.ProxyPassword, data.ProxyUri);
                        else
                            options.Proxy = new ProxySettings(data.ProxyUri, data.UseDefaultCredentials);
                    }
                    else if (data.ProxyUser != null && data.ProxyUser.Length > 0 && data.ProxyPassword != null && data.ProxyPassword.Length > 0)
                        options.Proxy = new ProxySettings(data.ProxyUser, data.ProxyPassword);
                    configuration.ExtendedOptions = options;
                }
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                licenseManager.Initialize(configuration);
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        async void OnSettings()
        {
            try
            {
                var window = new ConfigWindow()
                {
                    DataContext = new ConfigWindowViewModel()
                };
                await window.ShowDialog(view);
                InitializeLicenseSpring();
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        async void OnProductDetails()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var details = licenseManager.GetProductDetails();
                if (details.AuthorizationType == AuthorizationMethod.KeyBased)
                {
                    IsKeyBased = true;
                    IsUserBased = false;
                }
                if (details.AuthorizationType == AuthorizationMethod.UserBased)
                {
                    IsUserBased = true;
                    IsKeyBased = false;
                }

                var window = new ProductDetailsWindow()
                {
                    DataContext = new ProductDetailsWindowViewModel(details)
                };
                await window.ShowDialog(view);
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        void OnActivateKeyBased()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if (Key != null && Key.Length > 0)
                    licenseManager.ActivateLicense(LicenseID.FromKey(Key));
                else
                    MsgBox.Show(_view, "Type key for activation", "Error", MsgBox.MessageBoxButtons.Ok);
                var license = licenseManager.CurrentLicense();
                if (license.IsValid())
                    ShowInfoScreen();
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }
        void OnActivateUserBased()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if ((User != null && User.Length > 0) || (Password != null && Password.Length > 0))
                    licenseManager.ActivateLicense(LicenseID.FromUser(User, Password));
                else
                    MsgBox.Show(_view, "Type credentials for activation", "Error", MsgBox.MessageBoxButtons.Ok);
                var license = licenseManager.CurrentLicense();
                if (license.IsValid())
                    ShowInfoScreen();
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        public async Task<string> SaveFile()
        {
            SaveFileDialog dialog = new SaveFileDialog();
            dialog.Filters.Add(new FileDialogFilter() { Name = "req", Extensions = { "req" } });

            string result = await dialog.ShowAsync(view);

            return string.Join(" ", result);
        }

        async void OnActivateKeyBasedOffline()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if (Key != null && Key.Length > 0)
                {
                    string path = await SaveFile();
                    licenseManager.GetOfflineActivationFile(LicenseID.FromKey(Key), path);
                }
                else
                    await MsgBox.Show(_view, "Type key for activation", "Error", MsgBox.MessageBoxButtons.Ok);

            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }
        async void OnActivateUserBasedOffline()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if ((User != null && User.Length > 0) || (Password != null && Password.Length > 0))
                {
                    string path = await SaveFile();
                    licenseManager.GetOfflineActivationFile(LicenseID.FromUser(User, Password), path);
                }
                else
                    await MsgBox.Show(_view, "Type credentials for activation", "Error", MsgBox.MessageBoxButtons.Ok);
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        public async Task<string> OpenFile()
        {
            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Filters.Add(new FileDialogFilter() { Name = "lic", Extensions = { "lic" } });

            string[] result = await dialog.ShowAsync(view);

            return string.Join(" ", result);
        }
        async void OnActivateOffline()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                LicensePath = await OpenFile();
                licenseManager.ActivateLicenseOffline(LicensePath);
                var license = licenseManager.CurrentLicense();
                if (license.IsValid())
                    ShowInfoScreen();
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        void OnOpenActivationURL()
        {
            try
            {
                ProcessStartInfo info = new ProcessStartInfo(@"C:\Program Files\Internet Explorer\iexplore.exe");
                info.Arguments = configuration.ExtendedOptions.AlternateServiceURL.Replace("api", "saas") + "offline/";
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                Process.Start(info);
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        void OnGetTrialLicense()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var licenseId = licenseManager.GetTrialLicense(User);
                if (licenseId.IsEmpty())
                {
                    MsgBox.Show(_view, "Got empty trial license!", "Error", MsgBox.MessageBoxButtons.Ok);
                    return;
                }
                if (string.IsNullOrEmpty(licenseId.User))
                    Key = licenseId.Key;
                else
                {
                    User = licenseId.User;
                    Password = licenseId.Password;
                }
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        private void HandleException(Exception ex)
        {
            if (ex is LicenseSpringException)
                MsgBox.Show(_view, "LicenseSpring exception occurred\n" + ex.Message, "Error", MsgBox.MessageBoxButtons.Ok);
            else
                MsgBox.Show(_view, "Exception occured\n" + ex.Message, "Error", MsgBox.MessageBoxButtons.Ok);
        }

        private ConfigData readConfigData()
        {
            ConfigData data;
            var licenseFolderPath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
            licenseFolderPath = Path.Combine(licenseFolderPath, "LicenseSpring");
            string fileName = Path.Combine(licenseFolderPath, "Settings.json");
            if (!File.Exists(fileName))
            {
                return null;
            }
            using (StreamReader r = new StreamReader(fileName))
            {
                string json = r.ReadToEnd();
                data = JsonSerializer.Deserialize<ConfigData>(json);
            }
            return data;
        }

    }
}
