﻿using System;
using System.IO;
using System.Windows.Forms;
using LicenseSpring;

namespace SSOSample
{
    public partial class InfoScreen : Form
    {
        public InfoScreen(Configuration configuration)
        {
            this.configuration = configuration;
            InitializeComponent();
            InitLists();
            UpdateLicenseInfo();
        }

        private void HandleException(Exception ex)
        {
            if (ex is LicenseSpringException)
                MessageBox.Show(ex.Message, "LicenseSpring exception occurred");
            else
                MessageBox.Show(ex.Message, "Exception occured");
        }

        private LicenseFeature[] features;
        private void UpdateLicenseInfo()
        {
            try
            {
                var licenseManager = LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                textBoxKey.Text = license.Key();
                textBoxUser.Text = license.User();
                textBoxType.Text = license.Type().ToString();
                textBoxStatus.Text = license.Status().ToString();
                textBoxIsOfflineActivated.Text = license.IsOfflineActivated().ToString();
                textBoxIsTrial.Text = license.IsTrial().ToString();
                textBoxIsValid.Text = license.IsValid().ToString();
                textBoxIsFloating.Text = license.IsFloating().ToString();
                textBoxStartDate.Text = license.StartDate().ToString("dd/MM/yyyy");
                textBoxValidityPeriod.Text = license.ValidityPeriod().ToString("dd/MM/yyyy");
                textBoxDays.Text = license.DaysRemaining().ToString();
                textBoxMaintenancePeriod.Text = license.MaintenancePeriod().ToString("dd/MM/yyyy");
                textBoxLastChecked.Text = license.LastCheckDate().ToString();
                textBoxTotalCount.Text = license.TotalConsumption().ToString();
                textBoxMaxCount.Text = license.MaxConsumption().ToString();
                textBoxFloatingTimeout.Text = license.FloatingTimeout().ToString();
                textBoxCurrentAppVersion.Text = (null != configuration) ? configuration.AppVersion : "";
                textBoxInfo.Text = Path.GetFullPath(Path.Combine(licenseManager.DataLocation,
                    licenseManager.LicenseFileName));


                listViewCustomDataFields.Items.Clear();
                var fields = license.CustomFields();
                foreach (var field in fields)
                {
                    ListViewItem item = new ListViewItem(field.Name);
                    item.SubItems.Add(field.Value);
                    listViewCustomDataFields.Items.Add(item);
                }

                UpdateLicenseFeaturesTable();
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        private void InitLists()
        {
            int columnCount = 3;
            var widthLVIF = listViewInstallationFile.ClientRectangle.Width;
            listViewInstallationFile.Columns.Add("Version", widthLVIF / columnCount);
            listViewInstallationFile.Columns.Add("Url", widthLVIF / columnCount);
            listViewInstallationFile.Columns.Add("Hash", widthLVIF / columnCount);

            columnCount = 2;
            var widthLVCDF = listViewCustomDataFields.ClientRectangle.Width;
            listViewCustomDataFields.Columns.Add("Name", widthLVCDF / columnCount);
            listViewCustomDataFields.Columns.Add("Value", widthLVCDF / columnCount);

            columnCount = 7;
            var widthLVLF = listViewLicenseFeatures.ClientRectangle.Width;
            listViewLicenseFeatures.Columns.Add("Code", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Name", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Feature Type", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Max Consumption", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Total Consumption", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Local Consumption", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Expiry Date", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Allow Overages", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Max Overages", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Reset Consumption", widthLVLF / columnCount);
            listViewLicenseFeatures.Columns.Add("Consumption Period", widthLVLF / columnCount);
        }

        private void buttonGetLicenseFilePath_Click(object sender, EventArgs e)
        {
            var warningMessage = "Couldn't open license file directory!";
            try
            {
                if (null == configuration)
                {
                    MessageBox.Show(warningMessage);
                    return;
                }
                var path = Path.GetFullPath(LicenseManager.GetInstance().DataLocation);
                if (0 == path.Length)
                {
                    MessageBox.Show(warningMessage);
                    return;
                }
                var dir = new DirectoryInfo(path);
                if (!dir.Exists)
                {
                    MessageBox.Show(warningMessage);
                    return;
                }
                System.Diagnostics.Process.Start(dir.FullName);
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        private void buttonClearLicenseFolder_Click(object sender, EventArgs e)
        {
            try
            {
                var licenseManager = LicenseManager.GetInstance();
                licenseManager.ClearLocalStorage();
                Close();
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        private void buttonCheckLicense_Click(object sender, EventArgs e)
        {
            try
            {
                var license = LicenseManager.GetInstance().CurrentLicense();
                if (null == license)
                {
                    MessageBox.Show("There is no license file!");
                    return;
                }
                license.LocalCheck();
                license.Check();
                MessageBox.Show("License successfully checked!");
                UpdateLicenseInfo();
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        private void buttonDeactivateLicense_Click(object sender, EventArgs e)
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                license.Deactivate();
                UpdateLicenseInfo();
            }
            catch (Exception ex)
            {
                if (ex is NetworkException)
                {
                    DialogResult result = MessageBox.Show(ex.Message + "\n Would you like to make request file for offline deactivation?", "Error", MessageBoxButtons.OKCancel);
                    if (result == System.Windows.Forms.DialogResult.OK)
                    {
                        try
                        {
                            SaveFileDialog fileDialog = new SaveFileDialog();
                            fileDialog.DefaultExt = "req";
                            fileDialog.AddExtension = true;
                            fileDialog.Filter = "Request Files(*.req)|*.req|All files(*.*)|*.*";
                            result = fileDialog.ShowDialog();
                            var path = (DialogResult.OK == result) ? fileDialog.FileName : "";
                            var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                            var license = licenseManager.CurrentLicense();
                            license.DeactivateOffline(path);
                            MessageBox.Show("Request path: " + path);
                        }
                        catch (Exception exception)
                        {
                            HandleException(exception);
                        }
                    }
                }
                else
                    HandleException(ex);
            }
        }

        private void buttonUpdateConsumption_Click(object sender, EventArgs e)
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                license.UpdateConsumption((int)numericUpdateConsumption.Value);
                textBoxTotalCount.Text = license.TotalConsumption().ToString();
                textBoxMaxCount.Text = license.MaxConsumption().ToString();
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        private void buttonSyncConsumption_Click(object sender, EventArgs e)
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                license.SyncConsumption();
                textBoxTotalCount.Text = license.TotalConsumption().ToString();
                textBoxMaxCount.Text = license.MaxConsumption().ToString();
                MessageBox.Show("Consumption synchronized!");
            }
            catch (Exception ex)
            {
                HandleException(ex);
            }
        }

        private void buttonLoadVersionList_Click( object sender, EventArgs e )
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();

                string[] versions = licenseManager.GetAllVersions( license.Id() );

                listViewInstallationFile.Items.Clear();
                foreach( var version in versions )
                {
                    InstallationFile installFile  = licenseManager.GetInstallationFile( license.Id(),
                        new InstallationFileOptions( version ) );
                    ListViewItem item = new ListViewItem( installFile.Version );
                    item.SubItems.Add( installFile.Url );
                    item.SubItems.Add( installFile.Md5Hash );
                    listViewInstallationFile.Items.Add( item );
                }
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void buttonCheckForUpdates_Click( object sender, EventArgs e )
        {
            try
            {
                var licenseManager = LicenseManager.GetInstance();
                var license = licenseManager.CurrentLicense();
                InstallationFile installFile = licenseManager.GetInstallationFile( license.Id() );

                if( string.IsNullOrEmpty( installFile.Version ) )
                {
                    MessageBox.Show("Couldn't get version from the server!");
                    return;
                }

                Version currentVer = new Version( configuration.AppVersion );
                Version version = new Version( installFile.Version );
                if( currentVer >= version )
                    MessageBox.Show( "The application is app to date." );
                else
                    MessageBox.Show( "There is a new version of the app available!" );
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void InfoScreen_FormClosing(object sender, FormClosingEventArgs e)
        {
            if( LicenseManager.GetInstance().CurrentLicense() != null )
                Application.Exit();
        }

        private readonly Configuration configuration = null;

        void UpdateLicenseFeaturesTable()
        {
            var licenseManager = LicenseManager.GetInstance();
            var license = licenseManager.CurrentLicense();
            features = license.Features();
            listViewLicenseFeatures.Items.Clear();
            foreach (var feature in features)
            {
                ListViewItem item = new ListViewItem(feature.Code);
                item.SubItems.Add(feature.Name);
                item.SubItems.Add(feature.FeatureType.ToString());
                item.SubItems.Add(feature.MaxConsumption.ToString());
                item.SubItems.Add(feature.TotalConsumption.ToString());
                item.SubItems.Add(feature.LocalConsumption.ToString());
                item.SubItems.Add(feature.ExpiryDate.ToString("dd/MM/yyyy"));
                item.SubItems.Add(feature.AllowOverages.ToString());
                item.SubItems.Add(feature.MaxOverages.ToString());
                item.SubItems.Add(feature.ResetConsumption.ToString());
                item.SubItems.Add(feature.ConsumptionPeriod.ToString());
                listViewLicenseFeatures.Items.Add(item);
            }
        }

        private void buttonSyncFeatureConsumption_Click(object sender, EventArgs e)
        {
            try
            {
                if (listViewLicenseFeatures.SelectedIndices.Count > 0)
                {
                    var items = listViewLicenseFeatures.SelectedIndices;
                    var licenseManager = LicenseManager.GetInstance();
                    var license = licenseManager.CurrentLicense();
                    foreach (var item in items)
                    {
                        license.SyncFeatureConsumption(features[(int)item].Code);
                    }
                    UpdateLicenseFeaturesTable();
                }
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        private void buttonUpdateFeatureConsumption_Click(object sender, EventArgs e)
        {
            try
            {
                if (listViewLicenseFeatures.SelectedIndices.Count > 0)
                {
                    var items = listViewLicenseFeatures.SelectedIndices;
                    var licenseManager = LicenseManager.GetInstance();
                    var license = licenseManager.CurrentLicense();

                    foreach (var item in items)
                    {
                        license.UpdateFeatureConsumption(features[(int)item].Code, (int)numericIncreaseConsumption.Value);
                    }
                    UpdateLicenseFeaturesTable();
                }
            }
            catch (Exception ex)
            {
                HandleException(ex);
                return;
            }
        }

        private void buttonProductDetails_Click(object sender, EventArgs e)
        {
            var licenseManager = LicenseManager.GetInstance();
            var details = licenseManager.GetProductDetails();
            var productDetails = new ProductDetailsScreen(details);
            productDetails.ShowDialog();
        }
    }
}
