﻿using System;
using System.Diagnostics;
using System.Windows.Forms;
using LicenseSpring;

namespace WinFormSample
{
    public partial class ActivationScreen : Form
    {
        public ActivationScreen()
        {
            InitializeComponent();
            InitializeLicenseSpring();
            TryToStartInfoScreen();
        }

        private void HandleException( Exception ex )
        {
            if( ex is LicenseSpringException )
                MessageBox.Show( ex.Message, "LicenseSpring exception occurred" );
            else
                MessageBox.Show( ex.Message, "Exception occured" );
        }

        private void ShowInfoScreen()
        {
            Hide();
            var infoScreen = new InfoScreen( configuration );
            infoScreen.ShowDialog();
            Show();
        }

        private void TryToStartInfoScreen()
        {
            var licenseManager = LicenseManager.GetInstance();
            if( licenseManager == null )
                return;
            var license = licenseManager.CurrentLicense();
            if( license == null )
                return;

            try
            {
                license.LocalCheck();
                if( license.IsFloating() )
                    SetWatchdog();
            }
            catch( FloatingTimeoutExpiredException )
            {
                try
                {
                    license.Register();
                    SetWatchdog();
                }
                catch( Exception ex )
                {
                    HandleException( ex );
                    if( ex is MaxFloatingReachedException )
                        Application.Exit();// application cannot use this license at the moment because floating license limit reached
                }
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }

            ShowInfoScreen();
        }

        private void InitializeLicenseSpring()
        {
            try
            {
                configuration = LSConfig.CreateConfiguration();
                var licenseManager = LicenseManager.GetInstance();
                licenseManager.Initialize( configuration );
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private Configuration configuration = null;

        private void activateKeyBasedButton_Click( object sender, EventArgs e )
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if( keyTextBox.MaskCompleted )
                    licenseManager.ActivateLicense( LicenseID.FromKey( keyTextBox.Text ) );
                else
                {
                    MessageBox.Show( "Type key for activation" );
                    return;
                }
                var license = licenseManager.CurrentLicense();
                if( license.IsFloating() )
                {
                    license.Register();
                    SetWatchdog();
                }
                if( license.IsValid() )
                {
                    ShowInfoScreen();
                }
            }
            catch( Exception ex )
            {
                HandleException( ex );
                return;
            }
        }

        private void activateUserBasedButton_Click( object sender, EventArgs e )
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if( userTextBox.Text.Length > 0 && passwordTextBox.Text.Length > 0 )
                    licenseManager.ActivateLicense( LicenseID.FromUser( userTextBox.Text, passwordTextBox.Text ) );
                else
                {
                    MessageBox.Show( "Type credentials for activation" );
                    return;
                }
                var license = licenseManager.CurrentLicense();
                if( license.IsFloating() )
                {
                    license.Register();
                    SetWatchdog();
                }
                if( license.IsValid() )
                {
                    ShowInfoScreen();
                }
            }
            catch( Exception ex )
            {
                HandleException( ex );
                return;
            }
        }

        private void getTrialKeyButton_Click( object sender, EventArgs e )
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var licenseId = licenseManager.GetTrialLicense( userTextBox2.Text );
                if( licenseId.IsEmpty() )
                {
                    MessageBox.Show( "Empty trial license!" );
                    return;
                }
                if( string.IsNullOrEmpty( licenseId.User ) )
                    keyTextBox.Text = licenseId.Key;
                else
                {
                    userTextBox.Text = licenseId.User;
                    passwordTextBox.Text = licenseId.Password;
                }
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void activateOfflineButton_Click( object sender, EventArgs e )
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                OpenFileDialog fileDialog = new OpenFileDialog();
                fileDialog.DefaultExt = "lic";
                fileDialog.AddExtension = true;
                fileDialog.Filter = "License files(*.lic)|*.lic|All files(*.*)|*.*";
                var result = fileDialog.ShowDialog();
                if( DialogResult.OK == result )
                {
                    licenseManager.ActivateLicenseOffline( fileDialog.FileName );
                    var license = licenseManager.CurrentLicense();
                    if( license.IsValid() )
                        ShowInfoScreen();
                }
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void createRequestKeyBasedButton_Click( object sender, EventArgs e )
        {
            try
            {
                if( keyTextBox.MaskCompleted )
                {
                    SaveFileDialog fileDialog = new SaveFileDialog();
                    fileDialog.DefaultExt = "req";
                    fileDialog.AddExtension = true;
                    fileDialog.Filter = "Request Files(*.req)|*.req|All files(*.*)|*.*";
                    var result = fileDialog.ShowDialog();
                    var path = ( DialogResult.OK == result ) ? fileDialog.FileName : "";
                    var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                    licenseManager.GetOfflineActivationFile( LicenseID.FromKey( keyTextBox.Text ), path );
                }
                else
                    MessageBox.Show( "Type key for activation" );
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void createRequestUserBasedButton_Click( object sender, EventArgs e )
        {
            try
            {
                if( userTextBox.Text.Length > 0 && passwordTextBox.Text.Length > 0 )
                {
                    SaveFileDialog fileDialog = new SaveFileDialog();
                    fileDialog.DefaultExt = "req";
                    fileDialog.AddExtension = true;
                    fileDialog.Filter = "Request Files(*.req)|*.req|All files(*.*)|*.*";
                    var result = fileDialog.ShowDialog();
                    var path = ( DialogResult.OK == result ) ? fileDialog.FileName : "";
                    var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                    licenseManager.GetOfflineActivationFile( LicenseID.FromUser( userTextBox.Text, passwordTextBox.Text ), path );

                }
                else
                    MessageBox.Show( "Type credentials for activation" );
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void openPageButton_Click( object sender, EventArgs e )
        {
            try
            {
                ProcessStartInfo info = new ProcessStartInfo( @"C:\Program Files\Internet Explorer\iexplore.exe" );
                info.Arguments = "https://saas.licensespring.com/offline/";
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                Process.Start( info );
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void buttonProductDetails_Click( object sender, EventArgs e )
        {
            try
            {
                var licenseManager = LicenseManager.GetInstance();
                var details = licenseManager.GetProductDetails();
                var infoScreen = new ProductDetailsScreen( details );
                infoScreen.ShowDialog();
                if( details.AuthorizationType == AuthorizationMethod.UserBased )
                {
                    userTextBox.Enabled = true;
                    passwordTextBox.Enabled = true;
                    activateUserBasedButton.Enabled = true;
                    createRequestUserBasedButton.Enabled = true;

                    keyTextBox.Enabled = false;
                    activateKeyBasedButton.Enabled = false;
                    createRequestKeyBasedButton.Enabled = false;
                }
                else if( details.AuthorizationType == AuthorizationMethod.KeyBased )
                {

                    keyTextBox.Enabled = true;
                    activateKeyBasedButton.Enabled = true;
                    createRequestKeyBasedButton.Enabled = true;

                    userTextBox.Enabled = false;
                    passwordTextBox.Enabled = false;
                    activateUserBasedButton.Enabled = false;
                    createRequestUserBasedButton.Enabled = false;
                }
            }
            catch( Exception ex )
            {
                HandleException( ex );
            }
        }

        private void SetWatchdog()
        {
            var license = LicenseManager.GetInstance().CurrentLicense();
            LicenseWatchdogSettings settings = new LicenseWatchdogSettings( HandleErrors );
            settings.RunCallbackOnEveryCheck = true;
            settings.HandleNoInternet = true;
            settings.NoInternetTimeout = 30;
            license.SetupLicenseWatchdog( settings );
        }

        private void HandleErrors( LicenseSpringException ex )
        {
            var license = LicenseManager.GetInstance().CurrentLicense();
            if( ex == null )
            {
                MessageBox.Show( "License successfully checked: " + license.LastCheckDate() );
            }
            else
            {
                HandleException( ex );
                if( ex is MaxFloatingReachedException )
                    Application.Exit();//application cannot use this license at the moment because floating license limit reached
            }
        }
    }
}
