﻿using System;
using System.ComponentModel;
using System.IO;
using Xamarin.Forms;
using XamarinSample.Models;
using LicenseSpring;
using System.Windows.Input;
using System.Text.Json;
using XamarinSample.Views;
using Plugin.DeviceInfo;

namespace XamarinSample.ViewModels
{
    class MainPageViewModel : INotifyPropertyChanged
    {
        private static Page page;
        public Page Page
        {
            set
            {
                page = value;
            }
            get
            {
                return page;
            }
        }
        public event PropertyChangedEventHandler PropertyChanged;
        public MainPageModel Model { get; private set; }
        public string Key
        {
            get { return Model.Key; }
            set
            {
                if (Model.Key != value)
                {
                    Model.Key = value;
                    OnPropertyChanged("Key");
                }
            }
        }
        public string User
        {
            get { return Model.User; }
            set
            {
                if (Model.User != value)
                {
                    Model.User = value;
                    OnPropertyChanged("User");
                }
            }
        }
        public string Password
        {
            get { return Model.Password; }
            set
            {
                if (Model.Password != value)
                {
                    Model.Password = value;
                    OnPropertyChanged("Password");
                }
            }
        }
        public string FilePath
        {
            get { return Model.FilePath; }
            set
            {
                if (Model.FilePath != value)
                {
                    Model.FilePath = value;
                    OnPropertyChanged("Path");
                }
            }
        }
        public bool IsKeyBased
        {
            get { return Model.IsKeyBased; }
            set
            {
                if (Model.IsKeyBased != value)
                {
                    Model.IsKeyBased = value;
                    OnPropertyChanged("IsKeyBased");
                }
            }
        }
        public bool IsUserBased
        {
            get { return Model.IsUserBased; }
            set
            {
                if (Model.IsUserBased != value)
                {
                    Model.IsUserBased = value;
                    OnPropertyChanged("IsUserBased");
                }
            }
        }
        public ICommand ActivateKeyBased { private set; get; }
        public ICommand ActivateUserBased { private set; get; }
        public ICommand ActivateTrial { private set; get; }
        public ICommand ProductDetails { private set; get; }
        public ICommand GetTrialLicense { private set; get; }
        public ICommand Settings { private set; get; }
        public MainPageViewModel()
        {
            Model = new MainPageModel();
            IsKeyBased = true;
            IsUserBased = true;
            ActivateKeyBased = new Command(OnActivateKeyBased);
            ActivateUserBased = new Command(OnActivateUserBased);
            GetTrialLicense = new Command(OnGetTrialLicense);
            Settings = new Command(OnSettings);
            ProductDetails = new Command(OnProductDetails);
        }
        public void OnShow()
        {
            Page.Appearing += (s, e) => InitializeLicenseSpring();
            Page.Appearing += (s, e) => tryToStartInfoScreen();
        }
        protected void OnPropertyChanged(string propName)
        {
            if (PropertyChanged != null)
                PropertyChanged(this, new PropertyChangedEventArgs(propName));
        }
        private void tryToStartInfoScreen()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if (licenseManager == null)
                    return;
                var license = licenseManager.CurrentLicense();
                if (license == null)
                    return;
                if (license.IsValid())
                {
                    license.LocalCheck();
                    showInfoScreen();
                }
            }
            catch (Exception) { }
        }
        private void showInfoScreen()
        {
            Device.BeginInvokeOnMainThread(async () => await Page.Navigation.PushAsync(new InfoTabbedPage()
            {
                BindingContext = new InfoPageViewModel(Model.Configuration)
            }));
        }
     
        public void InitializeLicenseSpring()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if( licenseManager.IsInitialized() )
                    return;
                Model.Configuration = LSConfig.CreateConfiguration();
                ExtendedOptions options = Model.Configuration.ExtendedOptions;
                options.HardwareID = App.PlatformData.GetDeviceId();
                ConfigData data = readConfigData();
                if (data != null)
                {
                    options.CollectNetworkInfo = data.CollectNetworkInfo;
                    options.EnableLogging = data.EnableLogging;
                }
                Model.Configuration.ExtendedOptions = options;
                licenseManager.Initialize(Model.Configuration);
            }
            catch (Exception ex)
            {
                handleException(ex);
                return;
            }
        }

        void OnSettings()
        {
            try
            {
                Device.BeginInvokeOnMainThread(
                    async () =>
                    {
                        await Page.Navigation.PushAsync(new ConfigPage()
                        {
                            BindingContext = new ConfigPageViewModel()
                        }); InitializeLicenseSpring();
                    });
            }
            catch (Exception ex)
            {
                handleException(ex);
                return;
            }
        }
        void OnProductDetails()
        {
            try
            {
                LicenseSpring.LicenseManager licenseManager;
                licenseManager = (LicenseSpring.LicenseManager)LicenseSpring.LicenseManager.GetInstance();
                ProductDetails details = licenseManager.GetProductDetails();
                if (details.AuthorizationType == AuthorizationMethod.KeyBased)
                {
                    IsKeyBased = true;
                    IsUserBased = false;
                }
                if (details.AuthorizationType == AuthorizationMethod.UserBased)
                {
                    IsUserBased = true;
                    IsKeyBased = false;
                }
                Device.BeginInvokeOnMainThread(async () => await Page.Navigation.PushAsync(new ProductDetailsPage()
                {
                    BindingContext = new ProductDetailsViewModel(details)
                }));
            }
            catch (Exception ex)
            {
                handleException(ex);
                return;
            }
        }
        void OnActivateKeyBased()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if (Key != null && Key.Length > 0)
                    licenseManager.ActivateLicense(LicenseID.FromKey(Key));
                else
                    Page.DisplayAlert("Error", "Type key for activation", "OK");
                var license = licenseManager.CurrentLicense();
                if (license.IsValid())
                    showInfoScreen();
            }
            catch (Exception ex)
            {
                handleException(ex);
                return;
            }
        }
        void OnActivateUserBased()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                if ((User != null && User.Length > 0) || (Password != null && Password.Length > 0))
                    licenseManager.ActivateLicense(LicenseID.FromUser(User, Password));
                else
                    Page.DisplayAlert("Error", "Type credentials for activation", "OK");
                var license = licenseManager.CurrentLicense();
                if (license.IsValid())
                    showInfoScreen();
            }
            catch (Exception ex)
            {
                handleException(ex);
                return;
            }
        }
        private ConfigData readConfigData()
        {
            ConfigData data;
            var licenseFolderPath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
            licenseFolderPath = Path.Combine(licenseFolderPath, "LicenseSpring");
            string fileName = Path.Combine(licenseFolderPath, "Settings.json");
            if (!File.Exists(fileName))
            {
                return null;
            }
            using (StreamReader r = new StreamReader(fileName))
            {
                string json = r.ReadToEnd();
                data = JsonSerializer.Deserialize<ConfigData>(json);
            }
            return data;
        }
        void OnGetTrialLicense()
        {
            try
            {
                var licenseManager = LicenseSpring.LicenseManager.GetInstance();
                var licenseId = licenseManager.GetTrialLicense(User);
                if (licenseId.IsEmpty())
                {
                    Device.BeginInvokeOnMainThread(async () => await Page.DisplayAlert("Error", "Got empty trial license!", "OK"));
                    return;
                }
                if (string.IsNullOrEmpty(licenseId.User))
                    Key = licenseId.Key;
                else
                {
                    User = licenseId.User;
                    Password = licenseId.Password;
                }
            }
            catch (Exception ex)
            {
                handleException(ex);
            }
        }
        private void handleException(Exception ex)
        {
            if (ex is LicenseSpringException)
                Device.BeginInvokeOnMainThread(async () => await Page.DisplayAlert("Error", "LicenseSpring exception occurred\n" + ex.Message, "OK"));
            else
                Device.BeginInvokeOnMainThread(async () => await Page.DisplayAlert("Error", "Exception occured\n" + ex.Message, "OK"));
        }
    }
}
