import os

from PyQt6.QtCore import Qt
from PyQt6.QtGui import QPixmap
from PyQt6.QtWidgets import (
    QFrame,
    QHBoxLayout,
    QLabel,
    QPushButton,
    QScrollArea,
    QVBoxLayout,
    QWidget,
)
from src.boxes import (
    BasicBox,
    CustomFieldsBox,
    InstallationFileBox,
    LicenseFeatureBox,
    LicenseInfoBox,
    LiceseFileBox,
    OfflineBox,
    TitleBox,
    TrialBox,
)
from src.widgets import DictionaryInfo, ExceptionPopUp

from licensespring.hardware import HardwareIdProviderSource
from licensespring.licensefile import License
from licensespring.licensefile.config import Configuration
from licensespring.licensefile.customer import Customer
from licensespring.licensefile.license_manager import LicenseID, LicenseManager


class LicenseWindow(QWidget):
    def __init__(self, license: License):
        super().__init__()
        self.setWindowTitle("License Window")
        self.setStyleSheet(
            """
            QWidget {
                background-color: #F9F7F7;
            }

            QScrollBar:vertical {
                background: transparent;
                width: 8px;
                margin: 0px;
            }

            QScrollBar::handle:vertical {
                background: rgba(0, 0, 0, 0.2);
                min-height: 20px;
                border-radius: 4px;
            }

            QScrollBar::handle:vertical:hover {
                background: rgba(0, 0, 0, 0.3);
            }

            QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {
                height: 0px;
                subcontrol-origin: margin;
            }

            QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical {
                background: none;
            }

            QScrollBar:horizontal {
                height: 0px;  /* Hide horizontal scrollbar */
            }
        """
        )
        self.setGeometry(600, 600, 900, 800)
        self.license = license

        # ========== Create scrollable content ==========
        content_widget = QWidget()
        content_layout = QHBoxLayout(content_widget)
        content_widget.setLayout(content_layout)

        vertical_layout_left = QVBoxLayout()
        vertical_layout_right = QVBoxLayout()

        # Populate left side
        self.licensefile_widget = LiceseFileBox(
            self.license.licensefile_handler.license_path
        )
        self.licensefile_widget.clear_data.connect(self.handle_clear_storage)

        self.license_info_widget = LicenseInfoBox(self.license)
        self.license_info_widget.license_action.connect(self.handle_license)

        vertical_layout_left.addWidget(self.licensefile_widget)
        vertical_layout_left.addWidget(self.license_info_widget)

        # Populate right side
        self.installation_file_widget = InstallationFileBox()
        self.license_feature_widget = LicenseFeatureBox(
            self.license.licensefile_handler._cache.feature_manager
        )
        self.license_feature_widget.feature_data.connect(self.handle_features)

        self.custom_field_widget = CustomFieldsBox(self.license.custom_fields())

        vertical_layout_right.addWidget(self.installation_file_widget)
        vertical_layout_right.addWidget(self.license_feature_widget)
        vertical_layout_right.addWidget(self.custom_field_widget)

        content_layout.addLayout(vertical_layout_left)
        content_layout.addLayout(vertical_layout_right)

        # ========== Add scroll area ==========
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setWidget(content_widget)
        scroll_area.setFrameShape(QFrame.Shape.NoFrame)

        layout = QVBoxLayout(self)
        layout.addWidget(scroll_area)
        self.setLayout(layout)

        self.show()

    def handle_features(self, data: dict):
        event = data.get("event")
        try:
            if event == "check":
                self.license.check_feature(data.get("code"))

            elif event == "add_local_consumption":
                self.license.add_local_feature_consumption(
                    data.get("code"), data.get("consumptions")
                )

            elif event == "sync_consumption":
                self.license.sync_feature_consumption(data.get("code"))

            new_feature_data = self.license.get_feature_data(data.get("code"))
            self.license_feature_widget.update_feature_data(new_feature_data)

        except Exception as ex:
            self.pop_up = ExceptionPopUp(str(ex))
            self.pop_up.show()

    def handle_license(self, data: dict):
        event = data.get("event")

        try:
            if event == "check":
                response = self.license.check()
                self.license_feature_widget.update_feature_list(
                    self.license.licensefile_handler._cache.feature_manager.attributes_to_list()
                )
                self.custom_field_widget.update_custom_field_list(
                    self.license.custom_fields()
                )
                self.installation_file_widget.update_installation_files(
                    {
                        "Version": response.get("version"),
                        "Installation File": response.get("installation_file"),
                        "Release Date": response.get("release_date"),
                        "hash md5": response.get("hash_md5"),
                        "Requires Version": response.get("requires_version"),
                        "Environment": response.get("environment"),
                        "Eula link": response.get("eula_link"),
                        "Release Notes link": response.get("release_notes_link"),
                        "Size": response.get("size"),
                        "Channel": response.get("channel"),
                    }
                )

            elif event == "add_local_consumption":
                self.license.add_local_consumption(data.get("consumptions"))

            elif event == "sync_consumption":
                self.license.sync_consumption()

            elif event == "deactivate":
                self.license.deactivate()

            new_license_data = self.license_info_widget.transform_data(self.license)
            self.license_info_widget.update_license_data(new_license_data)

        except Exception as ex:
            self.pop_up = ExceptionPopUp(str(ex))
            self.pop_up.show()

    def handle_clear_storage(self):
        self.license.licensefile_handler.clear_storage()
        self.close()
        conf = Configuration(
            product=os.getenv("PRODUCT_CODE"),
            api_key=os.getenv("API_KEY"),
            shared_key=os.getenv("SHARED_KEY"),
            file_key="d66db34b03c2d6961bb3e14ff40592c0c39ec7210113f194c0da50c2d4d002be",
            file_iv="a770af52b2aa3b73ad218b6cfc4e9707",
            api_domain=os.getenv("API_URL", "api.licensespring.com"),
            is_guard_file_enabled=True,
            verify_license_signature=False,
            hardware_id_provider=HardwareIdProviderSource,
        )
        license_manager = LicenseManager(conf)

        self.window = SampleApp(license_manager)
        self.window.show()

    def center_on_window(self, parent_window: QWidget):
        screen = parent_window.screen().geometry()
        self.move(
            screen.x() + (screen.width() - self.width()) // 2,
            screen.y() + (screen.height() - self.height()) // 2,
        )


class SampleApp(QWidget):
    def __init__(self, license_manager: LicenseManager):
        super().__init__()
        self.setStyleSheet("background-color: #F9F7F7;")
        self.license_manager = license_manager
        self.setWindowTitle("Python SDK Demo App")

        self.online_activation = TitleBox("Online activation")
        self.offline_activation = OfflineBox("Offline activation")

        self.offline_activation.path.connect(self.handle_offline_submit)

        # Main layout
        self.main_layout = QVBoxLayout()
        self.top_layout = QHBoxLayout()

        # Button
        button = QPushButton("Product Details")
        button.clicked.connect(self.get_product_details)
        button.setStyleSheet(
            """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 8px;
                padding: 10px 20px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
        )

        self.top_layout.addWidget(button, alignment=Qt.AlignmentFlag.AlignLeft)

        # Logo
        pixmap = QPixmap("images/ls-logo.png")
        label = QLabel()
        label.setPixmap(pixmap)
        self.main_layout.addWidget(label, alignment=Qt.AlignmentFlag.AlignHCenter)

        self.main_layout.addLayout(self.top_layout)

        self.key_based = BasicBox("Key-Based", 250, 250, "license-key")
        self.user_based = BasicBox("User-Based", 250, 250, "username", "password")
        self.trial_based = TrialBox("Trial", 250, 250, "email")

        self.online_activation.addWidgetHorizontal(self.key_based)
        self.online_activation.addWidgetHorizontal(self.user_based)
        self.online_activation.addWidgetHorizontal(self.trial_based)

        self.key_based.activation.connect(self.handle_activation)
        self.user_based.activation.connect(self.handle_activation)
        self.trial_based.activation.connect(self.handle_activation)

        self.key_based.create_request.connect(self.handle_create_offline_activation)
        self.user_based.create_request.connect(self.handle_create_offline_activation)

        self.main_layout.addWidget(self.online_activation)
        self.main_layout.addWidget(self.offline_activation)
        self.setLayout(self.main_layout)

    def handle_activation(self, data: dict):
        title = data.get("title")
        try:
            if title == "Key-Based":
                license_id = LicenseID.from_key(data.get("license-key", None))
                license = self.license_manager.activate_license(license_id=license_id)
            elif title == "User-Based":
                license_id = LicenseID.from_user(
                    username=data.get("username", None),
                    password=data.get("password", None),
                )
                license = self.license_manager.activate_license(license_id=license_id)
            else:
                license_id = self.license_manager.get_trial_license(
                    Customer(data.get("email", None))
                )
                license = self.license_manager.activate_license(license_id=license_id)

        except Exception as ex:
            self.pop_up = ExceptionPopUp(str(ex))
            self.pop_up.show()
            license = None

        if license:
            self.new_window = LicenseWindow(license)
            self.new_window.center_on_window(self)
            self.close()

    def handle_create_offline_activation(self, data: dict):
        title = data.get("title")
        try:
            if title == "Key-Based":
                license_id = LicenseID.from_key(data.get("license-key", None))
                self.license_manager.create_offline_activation_file(
                    license_id=license_id, req_path=data.get("file_path")
                )
            elif title == "User-Based":
                license_id = LicenseID.from_user(
                    username=data.get("username", None),
                    password=data.get("password", None),
                )
                self.license_manager.create_offline_activation_file(
                    license_id=license_id, req_path=data.get("file_path")
                )

        except Exception as ex:
            self.pop_up = ExceptionPopUp(str(ex))
            self.pop_up.show()

    def handle_offline_submit(self, path):
        try:
            license = self.license_manager.activate_license_offline(path)
            self.new_window = LicenseWindow(license)
            self.close()

        except Exception as ex:
            self.pop_up = ExceptionPopUp(str(ex))
            self.pop_up.show()

    def get_product_details(self):
        try:
            response = self.license_manager.get_product_details()
            self.details_window = DictionaryInfo(response)
            self.details_window.show()

        except Exception as ex:
            self.pop_up = ExceptionPopUp(str(ex))
            self.pop_up.show()
