import os
import platform
import subprocess
import webbrowser

from PyQt6.QtCore import Qt, pyqtSignal
from PyQt6.QtGui import QColor
from PyQt6.QtWidgets import (
    QGraphicsDropShadowEffect,
    QGroupBox,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QPushButton,
    QVBoxLayout,
    QWidget,
)
from src.tables import CustomFieldTable, FeatureTable
from src.widgets import (
    ConsumptionWidget,
    DictionaryInfo,
    FolderSelectWindow,
    LicensePathWindow,
)

from licensespring.licensefile import License


class TitleBox(QGroupBox):
    def __init__(self, title: str):
        super().__init__(title)

        self.main_layout = QVBoxLayout()
        self.setLayout(self.main_layout)

        self.group_layout_inputs = QVBoxLayout()
        self.main_layout.addLayout(self.group_layout_inputs)

        self.group_layout_buttons = QHBoxLayout()
        self.main_layout.addLayout(self.group_layout_buttons)

        self.setStyleSheet(
            """
                QGroupBox {
                border-radius: 4px;
                background-color: #DBE2EF;
                margin-top: 20px;
            }
            QGroupBox::title {
                subcontrol-origin: margin;
                color: #3F72AF;  /* Explicitly style title text */
                background-color: transparent;
                font: 18px "Arial";
                subcontrol-position: top center;
                padding: 10px
            }
            """
        )

        shadow = QGraphicsDropShadowEffect()
        shadow.setBlurRadius(30)
        shadow.setOffset(5, 5)
        shadow.setColor(QColor(63, 114, 175, 160))
        self.setGraphicsEffect(shadow)

        self.main_layout.setContentsMargins(15, 15, 15, 15)
        self.main_layout.setSpacing(10)

    def addWidgetVertical(self, widget: QWidget):
        """Add widget to the vertical layout."""
        self.group_layout_inputs.addWidget(widget)

    def addWidgetHorizontal(self, widget: QWidget):
        """Add widget to the horizontal layout."""
        self.group_layout_buttons.addWidget(widget)


# License File Box
class LiceseFileBox(QWidget):
    clear_data = pyqtSignal()

    def __init__(self, folder_path):
        super().__init__()
        self.folder_path = folder_path
        self.setFixedHeight(125)
        self.main_layout = QVBoxLayout()

        self.title_box = TitleBox("licesefile")

        # Add License file path
        self.label_key = QLabel(f"License file path:")
        self.label_key.setStyleSheet(
            """
            color: #112D4E;
            background-color: #DBE2EF;
        """
        )
        self.label_value = QLabel(str(self.folder_path))
        self.label_value.setStyleSheet(
            """
            color: #3F72AF;
            background-color: #DBE2EF;
        """
        )

        self.title_box.addWidgetHorizontal(self.label_key)
        self.title_box.addWidgetHorizontal(self.label_value)

        self.button_row = QHBoxLayout()

        self.open_folder_btn = QPushButton("Open license folder")
        self.open_folder_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
        )
        self.open_folder_btn.clicked.connect(self.open_license_folder)

        self.clear_data_btn = QPushButton("Clear license folder")
        self.clear_data_btn.setStyleSheet(
            """
            QPushButton {
                color: white;
                background-color: #B91C1C; /* red-700 */
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }

            QPushButton:hover {
                background-color: #991B1B; /* red-800 */
            }

            QPushButton:focus {
                outline: none;
                background-color: #991B1B;
            }

            QPushButton:pressed {
                background-color: #7F1D1D;
            }
            """
        )
        self.clear_data_btn.clicked.connect(self.clear_license_folder)

        self.button_row.addWidget(self.open_folder_btn)
        self.button_row.addWidget(self.clear_data_btn)
        self.title_box.main_layout.addLayout(self.button_row)
        self.main_layout.addWidget(self.title_box)

        self.setLayout(self.main_layout)

    def open_license_folder(self):
        if platform.system() == "Windows":
            os.startfile(self.folder_path)
        elif platform.system() == "Darwin":
            subprocess.run(["open", self.folder_path])
        else:
            subprocess.run(["xdg-open", self.folder_path])

    def clear_license_folder(self):
        self.clear_data.emit()


# License info Box
class LicenseInfoBox(QWidget):
    license_action = pyqtSignal(dict)

    def __init__(self, license: License):
        super().__init__()
        self.data = self.transform_data(license)
        self.main_layout = QVBoxLayout()
        self.title_box = TitleBox("License info")
        self.disp_data = DictionaryInfo(self.data)

        # button
        self.button_row = QHBoxLayout()

        self.check_button = QPushButton("Check")
        self.check_button.setStyleSheet(
            """
            QPushButton {
                background-color: #9DC08B;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #609966;
            }
            QPushButton:pressed {
                background-color: #609966;
            }
        """
        )
        self.check_button.clicked.connect(
            lambda: self.handle_license({"event": "check"})
        )

        self.deactivate_button = QPushButton("Deactivate")
        self.deactivate_button.setStyleSheet(
            """
            QPushButton {
                color: white;
                background-color: #B91C1C; /* red-700 */
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }

            QPushButton:hover {
                background-color: #991B1B; /* red-800 */
            }

            QPushButton:focus {
                outline: none;
                background-color: #991B1B;
            }

            QPushButton:pressed {
                background-color: #7F1D1D;
            }
            """
        )

        self.deactivate_button.clicked.connect(
            lambda: self.handle_license({"event": "deactivate"})
        )

        self.consumption_button = QPushButton("Consumption")
        self.consumption_button.setStyleSheet(
            """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
        )
        self.consumption_button.clicked.connect(self.handle_consumption)

        if license.license_type() != "consumption":
            self.consumption_button.setEnabled(False)

        self.button_row.addWidget(self.check_button)
        self.button_row.addWidget(self.deactivate_button)
        self.button_row.addWidget(self.consumption_button)

        self.title_box.addWidgetVertical(self.disp_data)

        self.title_box.main_layout.addLayout(self.button_row)

        self.main_layout.addWidget(self.title_box)

        self.setLayout(self.main_layout)

    def transform_data(self, license: License):
        data = {
            "License key": license.key(),
            "User": (
                license.license_user().get("user")
                if isinstance(license.license_user(), dict)
                else None
            ),
            "Status": "Active" if license.license_active() else "Inactive",
            "License Type": license.license_type(),
            "Is trial": license.is_trial(),
            "Is Valid": license.is_valid(),
            "Validity period": license.validity_period(),
            "Days remaining": license.days_remaining(),
            "Last Check": license.last_check(),
            "Maintenance period": license.maintenance_period(),
            "Is floating": license.is_floating(),
            "Floating timeout": license.floating_timeout(),
            "Total count": license.transfer_count(),
            "Max count": license.max_transfers(),
        }

        if license.license_type() == "consumption":
            data = {
                **data,
                "Total consumption": license.total_consumptions(),
                "Max consumption": license.max_consumptions(),
                "Local consumption": license.local_consumptions(),
            }

        return data

    def handle_license(self, data: dict):
        self.license_action.emit(data)

    def handle_consumption(self):
        self.consumption_window = ConsumptionWidget()
        self.consumption_window.local_consumption.connect(self.handle_license)
        self.consumption_window.sync.connect(self.handle_license)
        self.consumption_window.show()

    def update_license_data(self, data):
        self.disp_data.update_data(data)


class InstallationFileBox(QWidget):
    def __init__(self):
        super().__init__()
        self.main_layout = QVBoxLayout()

        self.title_box = TitleBox("Installation file")

        self.disp_data = DictionaryInfo(
            {
                "Version": "",
                "Installation File": "",
                "Release Date": "",
                "hash md5": "",
                "Requires Version": "",
                "Environment": "",
                "Eula link": "",
                "Release Notes link": "",
                "Size": "",
                "Channel": "",
            }
        )

        self.title_box.addWidgetVertical(self.disp_data)
        self.main_layout.addWidget(self.title_box)
        self.setLayout(self.main_layout)

    def update_installation_files(self, data: dict):
        self.disp_data.update_data(data)


class LicenseFeatureBox(QWidget):
    feature_data = pyqtSignal(dict)

    def __init__(self, features):
        super().__init__()
        self.main_layout = QVBoxLayout()
        self.title_box = TitleBox("License Features")
        self.scrollable_table = FeatureTable(features, ["Code", "Name", "Feature Type"])

        self.scrollable_table.feature_data.connect(self.handle_feature_event)

        self.title_box.addWidgetHorizontal(self.scrollable_table)
        self.main_layout.addWidget(self.title_box)
        self.setLayout(self.main_layout)

    def handle_feature_event(self, data):
        self.feature_data.emit(data)

    def update_feature_data(self, data: dict):
        self.scrollable_table.feature_window.disp_data.update_data(data)

    def update_feature_list(self, data: dict):
        if self.scrollable_table.data != data:
            self.scrollable_table.update_data(data)


class CustomFieldsBox(QWidget):
    def __init__(self, custom_fields):
        super().__init__()
        self.main_layout = QVBoxLayout()

        self.title_box = TitleBox("Custom data fields")
        self.custom_field_table = CustomFieldTable(custom_fields, ["Name", "Value"])
        self.title_box.addWidgetHorizontal(self.custom_field_table)
        self.main_layout.addWidget(self.title_box)
        self.setLayout(self.main_layout)

    def update_custom_field_list(self, data: dict):
        self.custom_field_table.update_data(data)


# Box for offline activation
class OfflineBox(TitleBox):
    path = pyqtSignal(str)

    def __init__(self, title: str):
        super().__init__(title)
        self.add_buttons()

    def add_buttons(self):
        self.open_page_btn = QPushButton("Open activation page")
        self.open_page_btn.setFixedSize(150, 40)
        self.open_page_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
        )

        self.activate_offline_btn = QPushButton("Activate offline")
        self.activate_offline_btn.setFixedSize(150, 40)
        self.activate_offline_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
        )

        self.open_page_btn.clicked.connect(self.open_page)
        self.activate_offline_btn.clicked.connect(self.activate_offline)

        self.addWidgetHorizontal(self.open_page_btn)
        self.addWidgetHorizontal(self.activate_offline_btn)
        self.group_layout_buttons.setAlignment(Qt.AlignmentFlag.AlignCenter)
        # self.group_layout_buttons.addStretch()

    def open_page(self):
        webbrowser.open(
            os.getenv("OFFLINE_PORTAL", "https://offline.licensespring.com")
        )

    def activate_offline(self):
        self.offline_activation_window = LicensePathWindow(
            "Please enter or select a .lic file path:", "Path to .lic file"
        )
        self.offline_activation_window.center_on_window(self)
        self.offline_activation_window.path.connect(self.send_path_data)
        self.offline_activation_window.show()

    def send_path_data(self, file_path):
        self.path.emit(file_path)


# Basic box -> user, key based activation
class BasicBox(QWidget):
    activation = pyqtSignal(dict)
    create_request = pyqtSignal(dict)

    def __init__(self, title: str, x_size, y_size, *args):
        super().__init__()

        self.title = title
        self.fields = args
        self.input_fields = {}

        self.setFixedSize(x_size, y_size)

        self.main_layout = QVBoxLayout()

        # Add title
        self.title_box = TitleBox(title)
        # Fields
        self.add_input_fields()
        # Buttons
        self.add_buttons()

        self.main_layout.addWidget(self.title_box)
        self.setLayout(self.main_layout)

    def add_input_fields(self):
        for field in self.fields:
            input_field = QLineEdit()
            input_field.setPlaceholderText(field)
            input_field.setStyleSheet(
                """
            QLineEdit {
                color: #112D4E;
                border: 1px solid #ccc;
                border-radius: 3px;
                padding: 6px 6px;
                font-size: 12px;
                background-color: #f9f9f9;
            }
            QLineEdit:focus {
                border: 1px solid #0078d7;
                background-color: #ffffff;
            }
        """
            )
            self.title_box.addWidgetVertical(input_field)
            self.input_fields[field] = input_field

    def add_buttons(self):
        # self.button_layout.setSpacing(5)
        self.activate_btn = QPushButton("Activate")
        self.activate_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #9DC08B;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #609966;
            }
            QPushButton:pressed {
                background-color: #609966;
            }
        """
        )

        self.create_req_btn = QPushButton("Create request")
        self.create_req_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
        )

        self.activate_btn.clicked.connect(self.handle_activate)
        self.create_req_btn.clicked.connect(self.handle_create_req)

        self.title_box.addWidgetHorizontal(self.activate_btn)
        self.title_box.addWidgetHorizontal(self.create_req_btn)

    def handle_activate(self):
        data = {key: field.text() for key, field in self.input_fields.items()}
        data = {**data, "title": self.title, "button": "activate"}
        self.activation.emit(data)

    def handle_create_req(self):
        self.select_folder = FolderSelectWindow(
            "Please enter or select the folder path where a activate_offline.req file will be created", "Folder path where the activate_offline.req file will be created"
        )
        self.select_folder.center_on_window(self)
        self.select_folder.path.connect(self.send_path_data)
        self.select_folder.show()

    def send_path_data(self, file_path):
        data = {key: field.text() for key, field in self.input_fields.items()}
        data = {
            **data,
            "title": self.title,
            "button": "create_request",
            "file_path": file_path,
        }
        self.create_request.emit(data)


class TrialBox(BasicBox):
    def __init__(self, title, x_size, y_size, *args):
        super().__init__(title, x_size, y_size, *args)

    def add_buttons(self):
        self.activate_btn = QPushButton("Get Trial")
        self.activate_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #9DC08B;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #609966;
            }
            QPushButton:pressed {
                background-color: #609966;
            }
        """
        )
        self.activate_btn.clicked.connect(self.handle_activate)
        self.title_box.addWidgetHorizontal(self.activate_btn)
