from PyQt6.QtCore import Qt, pyqtSignal
from PyQt6.QtWidgets import QHeaderView, QTableWidget, QTableWidgetItem
from src.widgets import FeatureDataWindow


class ScrollableTable(QTableWidget):
    def __init__(self, data: list, headers: list):
        super().__init__()
        self.data = data
        transformed_data = self.transform_list(data)
        self.setRowCount(len(transformed_data))
        self.setColumnCount(len(headers))
        self.setHorizontalHeaderLabels(headers)
        self.setSelectionBehavior(QTableWidget.SelectionBehavior.SelectRows)
        self.setFixedSize(350, 200)
        self.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeMode.Stretch)
        self.verticalHeader().setVisible(False)
        self.setShowGrid(False)

        self.setMouseTracking(True)
        self.hovered_row = -1

        self.setStyleSheet(
            """
            QTableWidget {
                border: none;
                font-size: 14px;
                font-family: Arial, sans-serif;
                background-color: #DBE2EF;
                gridline-color: #ddd;
            }
            QTableWidget::item {
                padding: 0px;
                border: none;
                background-color: #DBE2EF;
                color: #112D4E;
            }
            QTableWidget::item:selected {
                background-color: #3F72AF;
                color: white;
            }
            QHeaderView::section {
                background-color: #3F72AF;
                color: white;
                padding: 5px;
                border: 1px solid #ddd;
            }
            QTableWidget QTableCornerButton::section {
                background-color: #3F72AF;
            }
        """
        )

        for row_idx, row in enumerate(transformed_data):
            for col_idx, (key, value) in enumerate(row.items()):
                item = QTableWidgetItem(str(value))
                item.setFlags(item.flags() & ~Qt.ItemFlag.ItemIsEditable)
                item.setTextAlignment(Qt.AlignmentFlag.AlignCenter)
                self.setItem(row_idx, col_idx, item)

        self.cellClicked.connect(self.cell_was_clicked)

    def update_data(self, new_data):
        self.data = new_data  # Update the stored data
        transformed_data = self.transform_list(new_data)

        # Clear existing table
        self.setRowCount(0)
        self.setRowCount(len(transformed_data))

        for row_idx, row in enumerate(transformed_data):
            for col_idx, (key, value) in enumerate(row.items()):
                item = QTableWidgetItem(str(value))
                item.setFlags(item.flags() & ~Qt.ItemFlag.ItemIsEditable)
                item.setTextAlignment(Qt.AlignmentFlag.AlignCenter)
                self.setItem(row_idx, col_idx, item)

    def mouseMoveEvent(self, event):
        pos = event.pos()
        row = self.rowAt(pos.y())
        if row != self.hovered_row:
            self.hovered_row = row
            self.update_row_colors()
        super().mouseMoveEvent(event)

    def leaveEvent(self, event):
        self.hovered_row = -1
        self.update_row_colors()
        super().leaveEvent(event)

    def update_row_colors(self):
        for i in range(self.rowCount()):
            for j in range(self.columnCount()):
                item = self.item(i, j)
                if item:
                    if i == self.hovered_row:
                        item.setBackground(Qt.GlobalColor.transparent)
                        item.setBackground(Qt.GlobalColor.blue)
                        item.setForeground(Qt.GlobalColor.white)
                    elif item.isSelected():
                        item.setBackground(Qt.GlobalColor.transparent)
                    else:
                        item.setBackground(Qt.GlobalColor.transparent)
                        item.setBackground(Qt.GlobalColor.white)
                        item.setForeground(Qt.GlobalColor.black)

    def cell_was_clicked(self, row, column):
        pass

    def transform_list(self, data):
        raise Exception("Must be implemented")


class FeatureTable(ScrollableTable):
    feature_data = pyqtSignal(dict)

    def __init__(self, features, headers):
        self.features = features
        data = self.features.attributes_to_list()
        super().__init__(data, headers)

    def cell_was_clicked(self, row, column):
        code = [self.item(row, col).text() for col in range(self.columnCount())][0]
        self.feature_window = FeatureDataWindow(self.features.get_feature_dict(code))
        self.feature_window.feature_data.connect(self.handle_feature_event)

        self.feature_window.show()

    def transform_list(self, data):
        data_list = []

        for feature_dict in data:
            data_list.append(
                {
                    "Code": feature_dict.get("code", None),
                    "Name": feature_dict.get("name", None),
                    "Feature Type": feature_dict.get("feature_type", None),
                }
            )

        return data_list

    def handle_feature_event(self, data):
        self.feature_data.emit(data)


class CustomFieldTable(ScrollableTable):
    def __init__(self, data, headers):
        super().__init__(data, headers)

    def cell_was_clicked(self, row, column):
        pass

    def transform_list(self, data):
        data_list = []

        for custom_field_dict in data:
            data_list.append(
                {
                    "Name": custom_field_dict.get("name", None),
                    "Value": custom_field_dict.get("value", None),
                }
            )

        return data_list
