from PyQt6.QtCore import QPropertyAnimation, Qt, QTimer, pyqtSignal
from PyQt6.QtGui import QFont
from PyQt6.QtWidgets import (
    QApplication,
    QFileDialog,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QPushButton,
    QSlider,
    QSpinBox,
    QVBoxLayout,
    QWidget,
)


class ExceptionPopUp(QWidget):
    def __init__(self, message, duration=5000):
        super().__init__()
        self.setWindowFlags(
            Qt.WindowType.FramelessWindowHint
            | Qt.WindowType.WindowStaysOnTopHint
            | Qt.WindowType.Tool
        )
        self.setAttribute(Qt.WidgetAttribute.WA_TranslucentBackground)

        self.setStyleSheet(
            """
            QWidget {
                background-color: #F8D7DA;
                color: #721C24;
                border-radius: 10px;
            }
            QLabel {
                font-size: 14px;
            }
            QPushButton {
                background: transparent;
                border: none;
                font-size: 14px;
                font-weight: bold;
                color: #721C24;
                padding: 0px;
                margin: 0px;
            }
            QPushButton:hover {
                color: #A71D2A;
            }
        """
        )

        # Main layout for the popup
        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(10, 10, 10, 10)

        # Container widget to hold label and button in one box
        content_widget = QWidget()
        content_layout = QVBoxLayout(content_widget)
        content_layout.setContentsMargins(0, 0, 0, 0)
        content_layout.setSpacing(0)

        # Message + X in a single row
        label_layout = QHBoxLayout()
        label_layout.setContentsMargins(0, 0, 0, 0)

        message_label = QLabel(message)
        message_label.setWordWrap(True)
        message_label.setAlignment(
            Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignTop
        )

        close_btn = QPushButton("✖")
        close_btn.setFixedSize(20, 20)
        close_btn.setCursor(Qt.CursorShape.PointingHandCursor)
        close_btn.clicked.connect(self.close)

        label_layout.addWidget(message_label)
        label_layout.addStretch()
        label_layout.addWidget(close_btn, alignment=Qt.AlignmentFlag.AlignTop)

        content_layout.addLayout(label_layout)
        main_layout.addWidget(content_widget)

        QTimer.singleShot(duration, self.fade_out)

    def fade_out(self):
        self.animation = QPropertyAnimation(self, b"windowOpacity")
        self.animation.setDuration(1000)
        self.animation.setStartValue(1)
        self.animation.setEndValue(0)
        self.animation.finished.connect(self.close)
        self.animation.start()


class DictionaryInfo(QWidget):
    def __init__(self, data: dict):
        super().__init__()

        self.main_layout = QVBoxLayout()

        self.labels = {}
        for key, value in data.items():
            row_layout = QHBoxLayout()

            label_key = QLabel(f"{key}:")
            label_value = QLabel(str(value))

            label_key.setStyleSheet(
                """
                color: #112D4E;
                font-size: 14px;
            """
            )

            label_value.setStyleSheet(
                """
                color: #3F72AF;
                font-size: 14px;
            """
            )

            row_layout.addWidget(label_key)
            row_layout.addWidget(label_value, alignment=Qt.AlignmentFlag.AlignRight)

            self.main_layout.addLayout(row_layout)
            self.labels[key] = label_value

        self.setLayout(self.main_layout)
        self.setStyleSheet(
            """
            QWidget {
                background-color: #DBE2EF;
            }
        """
        )

    def update_data(self, new_data: dict):
        for key, value in new_data.items():
            if key in self.labels:
                self.labels[key].setText(str(value))


class ConsumptionWidget(QWidget):
    local_consumption = pyqtSignal(dict)
    sync = pyqtSignal(dict)

    def __init__(self):
        super().__init__()

        layout = QVBoxLayout(self)
        layout.setSpacing(20)

        slider_layout = QHBoxLayout()
        self.setStyleSheet("background-color: #DBE2EF")
        self.slider = QSlider(Qt.Orientation.Horizontal)
        self.slider.setMinimum(-100)
        self.slider.setMaximum(100)
        self.slider.setValue(0)
        self.slider.setStyleSheet(
            """
            QSlider::groove:horizontal {
                height: 8px;
                background: #767676;
                border-radius: 4px;
            }
            QSlider::handle:horizontal {
                background: #5cacee;
                width: 18px;
                border-radius: 9px;
                margin: -6px 0;
            }
        """
        )

        self.spin_box = QSpinBox()
        self.spin_box.setRange(-100, 100)
        self.spin_box.setValue(0)
        self.spin_box.setFixedHeight(40)
        self.spin_box.setStyleSheet(
            """
            QSpinBox {
                font-size: 18px;
                padding-right: 30px;
                border: 2px solid #5cacee;
                border-radius: 6px;
                background-color: #f0faff;
                color: #1c3f63;
            }

            QSpinBox::up-button, QSpinBox::down-button {
                background-color: #5cacee;
                border: none;
                width: 20px;
                height: 20px;
            }

            QSpinBox::up-button:hover, QSpinBox::down-button:hover {
                background-color: #4682b4;
            }
        """
        )

        # Sync slider and spinbox
        self.slider.valueChanged.connect(self.update_value)
        self.spin_box.valueChanged.connect(self.slider.setValue)

        slider_layout.addWidget(self.slider)
        slider_layout.addWidget(self.spin_box)
        layout.addLayout(slider_layout)

        # Buttons: Reset and Add 10
        button_layout = QHBoxLayout()
        self.sync_btn = QPushButton("Sync Consumption")
        self.add_btn = QPushButton("Add Consumption")

        for btn in [self.sync_btn, self.add_btn]:
            btn.setStyleSheet(
                """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
            )

        self.sync_btn.clicked.connect(self.sync_consumption)
        self.add_btn.clicked.connect(self.add_local_consumption)

        button_layout.addWidget(self.sync_btn)
        button_layout.addWidget(self.add_btn)
        layout.addLayout(button_layout)

    def update_value(self, value):
        self.spin_box.setValue(value)
        # self.result_label.setText(f"Current Value: {value}")

    def sync_consumption(self):
        self.sync.emit({"event": "sync_consumption"})

    def add_local_consumption(self):
        value = self.spin_box.value()
        self.local_consumption.emit(
            {"consumptions": int(value), "event": "add_local_consumption"}
        )
        self.slider.setValue(0)
        self.spin_box.setValue(0)


class FeatureDataWindow(QWidget):
    feature_data = pyqtSignal(dict)

    def __init__(self, data: dict) -> None:
        super().__init__()
        self.feature_code = data.get("code")
        self.setStyleSheet(
            """
            background-color: #DBE2EF;
        """
        )
        self.setWindowTitle("Feature View")
        self.main_layout = QVBoxLayout()
        self.button_layout = QVBoxLayout()

        # Add scrollable widgets to scroll_layout instead
        self.disp_data = DictionaryInfo(data)
        self.main_layout.addWidget(self.disp_data)

        if data.get("feature_type") == "consumption":
            self.consumption_window = ConsumptionWidget()
            self.consumption_window.local_consumption.connect(self.handle_feature_data)
            self.consumption_window.sync.connect(self.handle_feature_data)
            self.main_layout.addWidget(self.consumption_window)

        self.button = QPushButton("Feature Check")
        self.button.setStyleSheet(
            """
            QPushButton {
                background-color: #9DC08B;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #609966;
            }
            QPushButton:pressed {
                background-color: #609966;
            }
        """
        )
        self.button_layout = QHBoxLayout()
        self.button_layout.addWidget(self.button)

        self.main_layout.addLayout(self.button_layout)

        self.setLayout(self.main_layout)

        self.button.clicked.connect(
            lambda: self.handle_feature_data({"event": "check"})
        )

    def handle_feature_data(self, data: dict):
        feature_data = {"code": self.feature_code, **data}
        self.feature_data.emit(feature_data)


class PathWindow(QWidget):
    def __init__(self, title, path_input_holder):
        super().__init__()
        self.setGeometry(300, 300, 450, 120)
        self.setStyleSheet("background-color: #DBE2EF;")

        layout = QVBoxLayout()

        title = QLabel(title)
        title.setFont(QFont("Arial", 12))
        title.setStyleSheet("color: #112D4E; margin-bottom: 10px;")
        layout.addWidget(title)

        input_layout = QHBoxLayout()
        self.path_input = QLineEdit()

        self.path_input.setPlaceholderText(path_input_holder)
        self.path_input.setStyleSheet(
            """
            QLineEdit {
                color: #112D4E;
                border: 1px solid #ccc;
                border-radius: 3px;
                padding: 6px 6px;
                font-size: 12px;
                background-color: #f9f9f9;
            }
            QLineEdit:focus {
                border: 1px solid #0078d7;
                background-color: #ffffff;
            }
        """
        )
        input_layout.addWidget(self.path_input)

        browse_btn = QPushButton("Browse")
        browse_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #3F72AF;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #112D4E;
            }
            QPushButton:pressed {
                background-color: #112D4E;
            }
        """
        )

        self.confirm_btn = QPushButton("Confirm")
        self.confirm_btn.setStyleSheet(
            """
            QPushButton {
                background-color: #9DC08B;  /* Bootstrap blue */
                color: white;
                border: none;
                border-radius: 2px;
                padding: 5px 5px;
            }
            QPushButton:hover {
                background-color: #609966;
            }
            QPushButton:pressed {
                background-color: #609966;
            }
        """
        )
        self.confirm_btn.setEnabled(False)
        self.confirm_btn.clicked.connect(self.send_path)

        browse_btn.clicked.connect(self.browse)

        input_layout.addWidget(browse_btn)
        input_layout.addWidget(self.confirm_btn)

        layout.addLayout(input_layout)
        self.setLayout(layout)

    def browse(self):
        raise Exception("Needs to be implemented")

    def send_path(self):
        raise Exception("Needs to be implemented")

    def center_on_window(self, parent_window: QWidget):
        screen = parent_window.screen().geometry()
        self.move(
            screen.x() + (screen.width() - self.width()) // 2,
            screen.y() + (screen.height() - self.height()) // 2,
        )


class LicensePathWindow(PathWindow):
    path = pyqtSignal(str)

    def __init__(self, title, path_input_holder):
        super().__init__(title, path_input_holder)

    def browse(self):
        file_dialog = QFileDialog(
            self, "Select License File", "", "License Files (*.lic)"
        )
        file_dialog.setFileMode(QFileDialog.FileMode.ExistingFile)

        if file_dialog.exec():
            selected_files = file_dialog.selectedFiles()
            if selected_files:
                self.path_input.setText(selected_files[0])

        if self.path_input.text():
            self.confirm_btn.setEnabled(True)

    def send_path(self):
        self.path.emit(self.path_input.text())
        self.close()


class FolderSelectWindow(PathWindow):
    path = pyqtSignal(str)

    def __init__(self, title, path_input_holder):
        super().__init__(title, path_input_holder)

    def browse(self):
        dialog = QFileDialog(self, "Select Folder")
        dialog.setFileMode(QFileDialog.FileMode.Directory)
        dialog.setOption(QFileDialog.Option.ShowDirsOnly, True)

        if dialog.exec():
            selected_folders = dialog.selectedFiles()
            if selected_folders:
                self.path_input.setText(selected_folders[0])

        if self.path_input.text():
            self.confirm_btn.setEnabled(True)

    def send_path(self):
        self.path.emit(self.path_input.text())
        self.close()
